<?php
// telegram_api.php
require_once 'config.php';
require_once 'logger.php';

/**
 * اجرای درخواست به API تلگرام با استفاده از cURL
 */
function apiRequest($method, $parameters) {
    if (!is_string($method)) {
        logger()->error("نام متد باید رشته باشد", ['method' => $method]);
        return false;
    }
    if (!$parameters) {
        $parameters = [];
    } elseif (!is_array($parameters)) {
        logger()->error("پارامترها باید آرایه باشند", ['parameters' => $parameters]);
        return false;
    }
    
    // ثبت لاگ درخواست API
    logger()->debug("درخواست به API تلگرام", [
        'method' => $method,
        'parameters' => $parameters
    ]);
    
    $url = API_URL . $method;
    $handle = curl_init($url);
    curl_setopt($handle, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($handle, CURLOPT_CONNECTTIMEOUT, 5);
    curl_setopt($handle, CURLOPT_TIMEOUT, 60);
    
    // تنظیم هدرها برای UTF-8
    curl_setopt($handle, CURLOPT_HTTPHEADER, ["Content-Type: multipart/form-data; charset=UTF-8"]);
    
    // تبدیل پارامترها به UTF-8
    array_walk_recursive($parameters, function(&$item) {
        if (is_string($item)) {
            $item = mb_convert_encoding($item, 'UTF-8', 'UTF-8');
        }
    });
    
    curl_setopt($handle, CURLOPT_POSTFIELDS, $parameters);

    $response = curl_exec($handle);
    if ($response === false) {
        $error = curl_error($handle);
        logger()->error("خطای CURL در API تلگرام", [
            'method' => $method,
            'error' => $error
        ]);
        curl_close($handle);
        return false;
    }
    $httpCode = curl_getinfo($handle, CURLINFO_HTTP_CODE);
    curl_close($handle);

    if ($httpCode != 200) {
        logger()->error("خطا در درخواست API تلگرام", [
            'method' => $method,
            'http_code' => $httpCode,
            'response' => $response
        ]);
        return false;
    }
    
    $decodedResponse = json_decode($response, true);
    
    // ثبت لاگ پاسخ API
    logger()->debug("پاسخ از API تلگرام", [
        'method' => $method,
        'http_code' => $httpCode,
        'response' => $decodedResponse
    ]);
    
    return $decodedResponse;
}

/**
 * ارسال پیام متنی به کاربر
 */
function sendMessage($chat_id, $text, $reply_markup = null, $parse_mode = "HTML") {
    $parameters = [
        'chat_id' => $chat_id,
        'text'    => $text,
        'parse_mode' => $parse_mode,
    ];
    if ($reply_markup) {
        $parameters['reply_markup'] = json_encode($reply_markup);
    }
    
    // ثبت لاگ پیام ارسالی
    logger()->outgoingMessage($chat_id, $text, $reply_markup);
    
    return apiRequest('sendMessage', $parameters);
}

/**
 * ارسال سند (document)
 */
function sendDocument($chat_id, $document, $caption = '', $reply_markup = null) {
    $realPath = realpath($document);
    if (!$realPath) {
        logger()->error("فایل سند پیدا نشد", ['file_path' => $document]);
        return false;
    }
    
    // ثبت لاگ فایل ارسالی
    logger()->outgoingFile($chat_id, $document, 'document', $caption);
    
    $parameters = [
        'chat_id' => $chat_id,
        'caption' => $caption,
    ];
    if ($reply_markup) {
        $parameters['reply_markup'] = json_encode($reply_markup);
    }
    
    $url = API_URL . "sendDocument";
    $postFields = $parameters;
    $postFields['document'] = new CURLFile($realPath);
    
    $ch = curl_init(); 
    curl_setopt($ch, CURLOPT_HTTPHEADER, ["Content-Type: multipart/form-data"]);
    curl_setopt($ch, CURLOPT_URL, $url); 
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true); 
    curl_setopt($ch, CURLOPT_POSTFIELDS, $postFields); 
    $output = curl_exec($ch);
    if ($output === false) {
        $error = curl_error($ch);
        logger()->error("خطای ارسال سند", [
            'chat_id' => $chat_id,
            'file_path' => $document,
            'error' => $error
        ]);
    }
    curl_close($ch);
    
    return $output;
}

/**
 * ارسال ویدیو
 */
function sendVideo($chat_id, $video, $caption = '', $reply_markup = null) {
    $realPath = realpath($video);
    if (!$realPath) {
        logger()->error("فایل ویدیو پیدا نشد", ['file_path' => $video]);
        return false;
    }
    
    // ثبت لاگ فایل ارسالی
    logger()->outgoingFile($chat_id, $video, 'video', $caption);
    
    $parameters = [
        'chat_id' => $chat_id,
        'caption' => $caption,
    ];
    if ($reply_markup) {
        $parameters['reply_markup'] = json_encode($reply_markup);
    }
    
    $url = API_URL . "sendVideo";
    $postFields = $parameters;
    $postFields['video'] = new CURLFile($realPath);
    
    $ch = curl_init(); 
    curl_setopt($ch, CURLOPT_HTTPHEADER, ["Content-Type: multipart/form-data"]);
    curl_setopt($ch, CURLOPT_URL, $url); 
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true); 
    curl_setopt($ch, CURLOPT_POSTFIELDS, $postFields); 
    $output = curl_exec($ch);
    if ($output === false) {
        $error = curl_error($ch);
        logger()->error("خطای ارسال ویدیو", [
            'chat_id' => $chat_id,
            'file_path' => $video,
            'error' => $error
        ]);
    }
    curl_close($ch);
    
    return $output;
}

/**
 * ارسال صوت (audio)
 */
function sendAudio($chat_id, $audio, $caption = '', $reply_markup = null) {
    $realPath = realpath($audio);
    if (!$realPath) {
        logger()->error("فایل صوت پیدا نشد", ['file_path' => $audio]);
        return false;
    }
    
    // ثبت لاگ فایل ارسالی
    logger()->outgoingFile($chat_id, $audio, 'audio', $caption);
    
    $parameters = [
        'chat_id' => $chat_id,
        'caption' => $caption,
    ];
    if ($reply_markup) {
        $parameters['reply_markup'] = json_encode($reply_markup);
    }
    
    $url = API_URL . "sendAudio";
    $postFields = $parameters;
    $postFields['audio'] = new CURLFile($realPath);
    
    $ch = curl_init(); 
    curl_setopt($ch, CURLOPT_HTTPHEADER, ["Content-Type: multipart/form-data"]);
    curl_setopt($ch, CURLOPT_URL, $url); 
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true); 
    curl_setopt($ch, CURLOPT_POSTFIELDS, $postFields); 
    $output = curl_exec($ch);
    if ($output === false) {
        $error = curl_error($ch);
        logger()->error("خطای ارسال صوت", [
            'chat_id' => $chat_id,
            'file_path' => $audio,
            'error' => $error
        ]);
    }
    curl_close($ch);
    
    return $output;
}

function sendPhoto($chat_id, $photo_url, $caption = '', $reply_markup = null) {
    $data = [
        'chat_id' => $chat_id,
        'photo' => $photo_url,
        'caption' => $caption,
        'parse_mode' => 'Markdown'
    ];
    if ($reply_markup) {
        $data['reply_markup'] = json_encode($reply_markup);
    }
    apiRequest('sendPhoto', $data);
}
?>
