<?php
// logger.php - سیستم لاگ گیری کامل ربات تلگرام

/**
 * کلاس مدیریت لاگ‌ها
 * این کلاس امکان ثبت انواع مختلف لاگ‌ها را فراهم می‌کند
 */
class Logger {
    // انواع لاگ
    const ERROR = 'ERROR';
    const WARNING = 'WARNING';
    const INFO = 'INFO';
    const DEBUG = 'DEBUG';
    const USER_ACTION = 'USER_ACTION';
    const BOT_ACTION = 'BOT_ACTION';
    const CALLBACK = 'CALLBACK';
    
    // مسیر پوشه لاگ‌ها
    private $logDir;
    
    // فعال یا غیرفعال بودن لاگ‌گیری
    private $enabled = true;
    
    /**
     * سازنده کلاس Logger
     *
     * @param string $logDir مسیر پوشه لاگ‌ها
     */
    public function __construct($logDir = 'logs') {
        $this->logDir = $logDir;
        
        // ایجاد پوشه لاگ‌ها اگر وجود نداشته باشد
        if (!file_exists($this->logDir)) {
            mkdir($this->logDir, 0755, true);
        }
    }
    
    /**
     * فعال یا غیرفعال کردن لاگ‌گیری
     *
     * @param bool $enabled وضعیت فعال بودن
     */
    public function setEnabled($enabled) {
        $this->enabled = $enabled;
    }
    
    /**
     * ثبت لاگ
     *
     * @param string $level سطح لاگ
     * @param string $message پیام لاگ
     * @param array $context اطلاعات اضافی
     * @return bool نتیجه عملیات
     */
    public function log($level, $message, array $context = []) {
        if (!$this->enabled) {
            return false;
        }
        
        // ساخت نام فایل لاگ براساس تاریخ جاری
        $baseFilename = date('Y-m-d');
        $filename = $this->logDir . '/' . $baseFilename . '.log';
        
        // بررسی حجم فایل لاگ
        if (file_exists($filename) && filesize($filename) >= 1024 * 1024) { // 1MB
            // پیدا کردن شماره فایل جدید
            $counter = 1;
            while (file_exists($this->logDir . '/' . $baseFilename . '_' . $counter . '.log')) {
                $counter++;
            }
            
            // ایجاد فایل لاگ جدید
            $filename = $this->logDir . '/' . $baseFilename . '_' . $counter . '.log';
        }
        
        // ساخت رشته لاگ
        $logString = $this->formatLogMessage($level, $message, $context);
        
        // نوشتن لاگ در فایل
        return file_put_contents($filename, $logString, FILE_APPEND | LOCK_EX) !== false;
    }
    
    /**
     * قالب‌بندی پیام لاگ
     *
     * @param string $level سطح لاگ
     * @param string $message پیام لاگ
     * @param array $context اطلاعات اضافی
     * @return string رشته قالب‌بندی شده
     */
    private function formatLogMessage($level, $message, array $context = []) {
        // زمان فعلی با میکروثانیه
        $time = date('Y-m-d H:i:s') . '.' . sprintf('%03d', floor(microtime(true) * 1000) % 1000);
        
        // تبدیل اطلاعات اضافی به JSON
        $contextJson = !empty($context) ? json_encode($context, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT) : '';
        
        // ساخت رشته نهایی لاگ
        $logString = "[$time] [$level] $message";
        
        // اضافه کردن اطلاعات اضافی به رشته لاگ (اگر وجود داشته باشد)
        if (!empty($contextJson)) {
            $logString .= "\nContext: $contextJson";
        }
        
        return $logString . "\n\n";
    }
    
    /**
     * ثبت لاگ خطا
     *
     * @param string $message پیام خطا
     * @param array $context اطلاعات اضافی
     */
    public function error($message, array $context = []) {
        $this->log(self::ERROR, $message, $context);
    }
    
    /**
     * ثبت لاگ هشدار
     *
     * @param string $message پیام هشدار
     * @param array $context اطلاعات اضافی
     */
    public function warning($message, array $context = []) {
        $this->log(self::WARNING, $message, $context);
    }
    
    /**
     * ثبت لاگ اطلاعات
     *
     * @param string $message پیام اطلاعات
     * @param array $context اطلاعات اضافی
     */
    public function info($message, array $context = []) {
        $this->log(self::INFO, $message, $context);
    }
    
    /**
     * ثبت لاگ دیباگ
     *
     * @param string $message پیام دیباگ
     * @param array $context اطلاعات اضافی
     */
    public function debug($message, array $context = []) {
        $this->log(self::DEBUG, $message, $context);
    }
    
    /**
     * ثبت لاگ اقدام کاربر
     *
     * @param string $message توضیحات اقدام کاربر
     * @param array $context اطلاعات اضافی
     */
    public function userAction($message, array $context = []) {
        $this->log(self::USER_ACTION, $message, $context);
    }
    
    /**
     * ثبت لاگ اقدام ربات
     *
     * @param string $message توضیحات اقدام ربات
     * @param array $context اطلاعات اضافی
     */
    public function botAction($message, array $context = []) {
        $this->log(self::BOT_ACTION, $message, $context);
    }
    
    /**
     * ثبت لاگ کال‌بک
     *
     * @param string $message توضیحات کال‌بک
     * @param array $context اطلاعات اضافی
     */
    public function callback($message, array $context = []) {
        $this->log(self::CALLBACK, $message, $context);
    }
    
    /**
     * ثبت لاگ پیام دریافتی از کاربر
     *
     * @param array $message پیام دریافتی
     */
    public function incomingMessage($message) {
        $chat_id = $message['chat']['id'] ?? 'unknown';
        $username = $message['from']['username'] ?? 'unknown';
        $full_name = trim(($message['from']['first_name'] ?? '') . ' ' . ($message['from']['last_name'] ?? ''));
        
        $messageType = 'text';
        $messageContent = $message['text'] ?? '';
        
        // تشخیص نوع پیام (متن، عکس، فایل، و غیره)
        if (isset($message['document'])) {
            $messageType = 'document';
            $messageContent = 'File: ' . ($message['document']['file_name'] ?? 'unnamed file');
        } elseif (isset($message['photo'])) {
            $messageType = 'photo';
            $messageContent = 'Photo';
        } elseif (isset($message['video'])) {
            $messageType = 'video';
            $messageContent = 'Video';
        } elseif (isset($message['audio'])) {
            $messageType = 'audio';
            $messageContent = 'Audio: ' . ($message['audio']['title'] ?? 'unnamed audio');
        } elseif (isset($message['voice'])) {
            $messageType = 'voice';
            $messageContent = 'Voice message';
        } elseif (isset($message['sticker'])) {
            $messageType = 'sticker';
            $messageContent = 'Sticker';
        }
        
        $this->userAction("پیام دریافتی از کاربر", [
            'chat_id' => $chat_id,
            'username' => $username,
            'full_name' => $full_name,
            'message_type' => $messageType,
            'content' => $messageContent,
            'caption' => $message['caption'] ?? '',
            'raw_message' => $message
        ]);
    }
    
    /**
     * ثبت لاگ کال‌بک کوئری دریافتی
     *
     * @param array $callback کال‌بک کوئری
     */
    public function incomingCallback($callback) {
        $chat_id = $callback['message']['chat']['id'] ?? 'unknown';
        $username = $callback['from']['username'] ?? 'unknown';
        $full_name = trim(($callback['from']['first_name'] ?? '') . ' ' . ($callback['from']['last_name'] ?? ''));
        $data = $callback['data'] ?? 'unknown';
        
        $this->callback("کال‌بک دریافتی از کاربر", [
            'chat_id' => $chat_id,
            'username' => $username,
            'full_name' => $full_name,
            'callback_data' => $data,
            'raw_callback' => $callback
        ]);
    }
    
    /**
     * ثبت لاگ پیام ارسالی به کاربر
     *
     * @param int $chat_id شناسه چت
     * @param string $text متن پیام
     * @param array $markup کیبورد
     */
    public function outgoingMessage($chat_id, $text, $markup = null) {
        $this->botAction("پیام ارسالی به کاربر", [
            'chat_id' => $chat_id,
            'text' => $text,
            'markup' => $markup
        ]);
    }
    
    /**
     * ثبت لاگ فایل ارسالی به کاربر
     *
     * @param int $chat_id شناسه چت
     * @param string $file_path مسیر فایل
     * @param string $file_type نوع فایل
     * @param string $caption کپشن
     */
    public function outgoingFile($chat_id, $file_path, $file_type, $caption = '') {
        $this->botAction("فایل ارسالی به کاربر", [
            'chat_id' => $chat_id,
            'file_path' => $file_path,
            'file_type' => $file_type,
            'caption' => $caption
        ]);
    }
    
    /**
     * ثبت لاگ خطای اتصال به پایگاه داده
     *
     * @param \Exception $exception خطای رخ داده
     */
    public function databaseError(\Exception $exception) {
        $this->error("خطا در اتصال به پایگاه داده", [
            'message' => $exception->getMessage(),
            'code' => $exception->getCode(),
            'file' => $exception->getFile(),
            'line' => $exception->getLine(),
            'trace' => $exception->getTraceAsString()
        ]);
    }
    
    /**
     * ثبت لاگ خطای API تلگرام
     *
     * @param string $method متد API
     * @param array $parameters پارامترها
     * @param string $error پیام خطا
     */
    public function apiError($method, array $parameters, $error) {
        $this->error("خطا در API تلگرام", [
            'method' => $method,
            'parameters' => $parameters,
            'error' => $error
        ]);
    }
}

// ایجاد نمونه از کلاس Logger
$logger = new Logger();

/**
 * تابع کمکی برای دسترسی سریع به Logger
 *
 * @return Logger
 */
function logger() {
    global $logger;
    return $logger;
}

?> 