<?php
// exam_functions.php
require_once 'config.php';
require_once 'telegram_api.php';
require_once 'logger.php';

/**
 * نمایش منوی آزمون برای کاربران
 */
function showExamMenu($chat_id) {
    global $pdo;
    
    logger()->info("نمایش منوی آزمون برای کاربر", [
        'chat_id' => $chat_id,
        'function' => 'showExamMenu'
    ]);
    
    // بررسی آزمون‌های فعال
    $stmt = $pdo->prepare("SELECT id, title FROM exams WHERE start_time <= NOW() AND end_time >= NOW() ORDER BY start_time");
    $stmt->execute();
    $activeExams = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($activeExams)) {
        logger()->debug("هیچ آزمون فعالی یافت نشد", ['chat_id' => $chat_id]);
        sendMessage($chat_id, "در حال حاضر هیچ آزمون فعالی وجود ندارد.");
        return;
    }
    
    logger()->debug("آزمون‌های فعال یافت شد", [
        'chat_id' => $chat_id,
        'count' => count($activeExams),
        'exams' => $activeExams
    ]);
    
    $examButtons = [];
    foreach ($activeExams as $exam) {
        $examButtons[] = [
            ['text' => '📝 ' . $exam['title'], 'callback_data' => 'exam_start_' . $exam['id']]
        ];
    }
    
    $examButtons[] = [['text' => '🏠 بازگشت به منوی اصلی', 'callback_data' => 'exam_back_to_menu']];
    
    $reply_markup = ['inline_keyboard' => $examButtons];
    $examMenuText = "<b>🎯 آزمون‌های فعال</b>\n\n";
    $examMenuText .= "از لیست زیر یکی از آزمون‌های فعال را انتخاب کنید تا شرکت کنید یا نتایج را مشاهده نمایید.\n";
    $examMenuText .= "<i>هر آزمون دارای سوالات متنوع و قابلیت مشاهده رتبه است.</i>\n";
    sendMessage($chat_id, $examMenuText, $reply_markup, "HTML");
}

/**
 * مدیریت کال‌بک‌های مربوط به آزمون
 */
function handleExamCallback($callback) {
    global $pdo;
    
    $chat_id = $callback['message']['chat']['id'];
    $message_id = $callback['message']['message_id'];
    $data = $callback['data'];
    
    logger()->info("مدیریت کال‌بک آزمون", [
        'chat_id' => $chat_id,
        'callback_data' => $data,
        'function' => 'handleExamCallback'
    ]);
    
    // پاسخ اولیه به کال‌بک‌کوئری
    apiRequest('answerCallbackQuery', ['callback_query_id' => $callback['id']]);
    
    if ($data === 'exam_back_to_menu') {
        logger()->debug("بازگشت به منوی اصلی از منوی آزمون", ['chat_id' => $chat_id]);
        
        // ارسال منوی اصلی
        $welcomeText = "سلام و خوش آمدید به ربات آموزشی ما.\n\n" .
                       "لطفاً از منوی زیر یکی از دسته‌های آموزشی را انتخاب نمایید.";
        
        // واکشی دکمه‌های اصلی از پایگاه داده
        $stmt = $pdo->query("SELECT * FROM main_buttons ORDER BY title");
        $keyboardButtons = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $keyboardButtons[] = $row['title'];
        }
        
        // تقسیم دکمه‌ها به ردیف‌های چندتایی
        $keyboard = [];
        $row = [];
        $i = 0;
        foreach ($keyboardButtons as $btn) {
            $row[] = $btn;
            $i++;
            if ($i % 3 == 0) {
                $keyboard[] = $row;
                $row = [];
            }
        }
        if (!empty($row)) {
            $keyboard[] = $row;
        }
        
        $reply_markup = [
            'keyboard' => $keyboard,
            'resize_keyboard' => true,
            'one_time_keyboard' => true
        ];
        sendMessage($chat_id, $welcomeText, $reply_markup);
    }
    elseif (strpos($data, 'exam_start_') === 0) {
        $examId = str_replace('exam_start_', '', $data);
        logger()->info("کاربر آزمون را شروع کرد", [
            'chat_id' => $chat_id,
            'exam_id' => $examId
        ]);
        startExam($chat_id, $examId);
    }
    elseif (strpos($data, 'exam_question_') === 0) {
        // نمایش سوال آزمون
        // فرمت: exam_question_{exam_id}_{question_number}
        $params = str_replace('exam_question_', '', $data);
        $parts = explode('_', $params);
        
        if (count($parts) >= 2) {
            $examId = intval($parts[0]);
            $questionNumber = intval($parts[1]);
            
            logger()->info("نمایش سوال آزمون", [
                'chat_id' => $chat_id,
                'exam_id' => $examId,
                'question_number' => $questionNumber
            ]);
            
            showExamQuestion($chat_id, $examId, $questionNumber);
        }
    }
    elseif (strpos($data, 'exam_skip_') === 0) {
        // رد کردن سوال توسط کاربر
        // فرمت: exam_skip_{question_id}
        $questionId = intval(str_replace('exam_skip_', '', $data));
        logger()->info("رد کردن سوال توسط کاربر", [
            'chat_id' => $chat_id,
            'question_id' => $questionId
        ]);
        skipQuestion($chat_id, $questionId);
    }
    elseif (strpos($data, 'exam_answer_') === 0) {
        // ثبت پاسخ کاربر به سوال
        // فرمت: exam_answer_{question_id}_{option_id}
        $params = str_replace('exam_answer_', '', $data);
        $parts = explode('_', $params);
        
        if (count($parts) >= 2) {
            $questionId = intval($parts[0]);
            $optionId = intval($parts[1]);
            
            logger()->info("ثبت پاسخ کاربر به سوال", [
                'chat_id' => $chat_id,
                'question_id' => $questionId,
                'option_id' => $optionId
            ]);
            
            recordAnswer($chat_id, $questionId, $optionId);
        }
    }
    elseif (strpos($data, 'exam_finish_') === 0) {
        // پایان آزمون
        $examId = str_replace('exam_finish_', '', $data);
        
        logger()->info("کاربر آزمون را به پایان رساند", [
            'chat_id' => $chat_id,
            'exam_id' => $examId
        ]);
        
        finishExam($chat_id, $examId);
    }
}

/**
 * شروع آزمون برای کاربر
 */
function startExam($chat_id, $examId) {
    global $pdo;
    
    logger()->info("شروع آزمون برای کاربر", [
        'chat_id' => $chat_id,
        'exam_id' => $examId,
        'function' => 'startExam'
    ]);
    
    // بررسی وجود آزمون و فعال بودن آن
    $stmt = $pdo->prepare("SELECT * FROM exams WHERE id = :id AND start_time <= NOW() AND end_time >= NOW()");
    $stmt->execute([':id' => $examId]);
    $exam = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$exam) {
        logger()->warning("آزمون مورد نظر یافت نشد یا فعال نیست", [
            'chat_id' => $chat_id,
            'exam_id' => $examId
        ]);
        sendMessage($chat_id, "آزمون مورد نظر یافت نشد یا در حال حاضر فعال نیست.");
        return;
    }
    
    // بررسی اینکه آیا کاربر قبلاً در این آزمون شرکت کرده است
    $stmt = $pdo->prepare("
        SELECT * FROM results 
        WHERE exam_id = :exam_id AND user_id = (
            SELECT id FROM users WHERE telegram_id = :telegram_id
        )
    ");
    $stmt->execute([':exam_id' => $examId, ':telegram_id' => $chat_id]);
    $existingResult = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($existingResult) {
        logger()->warning("کاربر قبلاً در این آزمون شرکت کرده است", [
            'chat_id' => $chat_id,
            'exam_id' => $examId,
            'result_id' => $existingResult['id']
        ]);
        sendMessage($chat_id, "شما قبلاً در این آزمون شرکت کرده‌اید.");
        return;
    }
    
    // دریافت سوالات آزمون
    $stmt = $pdo->prepare("SELECT * FROM questions WHERE exam_id = :exam_id ORDER BY id");
    $stmt->execute([':exam_id' => $examId]);
    $questions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($questions)) {
        logger()->warning("آزمون هیچ سوالی ندارد", [
            'chat_id' => $chat_id,
            'exam_id' => $examId
        ]);
        sendMessage($chat_id, "این آزمون هیچ سوالی ندارد!");
        return;
    }
    
    logger()->debug("اطلاعات آزمون", [
        'exam_id' => $examId,
        'title' => $exam['title'],
        'questions_count' => count($questions)
    ]);
    
    // ارسال توضیحات آزمون و گزینه‌های شروع آن
    $examInfo = "*{$exam['title']}*\n\n";
    if (!empty($exam['description'])) {
        $examInfo .= "{$exam['description']}\n\n";
    }
    $examInfo .= "🕒 زمان پایان: " . date('Y-m-d H:i', strtotime($exam['end_time'])) . "\n";
    $examInfo .= "📝 تعداد سوالات: " . count($questions) . "\n\n";
    $examInfo .= "شما می‌توانید آزمون را در تلگرام یا از طریق وب انجام دهید.\nبرای تجربه بهتر، توصیه می‌شود از نسخه وب استفاده کنید.";
    
    // تولید لینک وب
    $webExamLink = generateWebExamLink($chat_id, $examId);
    
    $startButtons = [
        [['text' => '🚀 شروع آزمون در تلگرام', 'callback_data' => 'exam_question_' . $examId . '_1']],
        [['text' => '🌐 شروع آزمون در وب (توصیه می‌شود)', 'url' => $webExamLink]]
    ];
    
    $reply_markup = ['inline_keyboard' => $startButtons];
    sendMessage($chat_id, $examInfo, $reply_markup, "Markdown");
    
    // دریافت ID کاربر
    $stmt = $pdo->prepare("SELECT id FROM users WHERE telegram_id = :telegram_id");
    $stmt->execute([':telegram_id' => $chat_id]);
    $userId = $stmt->fetchColumn();
    
    // ثبت لاگ شروع آزمون
    $stmt = $pdo->prepare("
        INSERT INTO exam_logs (exam_id, user_id, action, details) 
        VALUES (:exam_id, :user_id, 'start', 'شروع آزمون')
    ");
    $stmt->execute([':exam_id' => $examId, ':user_id' => $userId]);
    
    logger()->info("لاگ شروع آزمون در پایگاه داده ثبت شد", [
        'chat_id' => $chat_id,
        'user_id' => $userId,
        'exam_id' => $examId
    ]);
}

/**
 * نمایش سوال آزمون
 */
function showExamQuestion($chat_id, $examId, $questionNumber) {
    global $pdo;
    
    logger()->info("نمایش سوال آزمون", [
        'chat_id' => $chat_id,
        'exam_id' => $examId,
        'question_number' => $questionNumber,
        'function' => 'showExamQuestion'
    ]);
    
    // بررسی وجود آزمون و فعال بودن آن
    $stmt = $pdo->prepare("SELECT * FROM exams WHERE id = :id AND end_time >= NOW()");
    $stmt->execute([':id' => $examId]);
    $exam = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$exam) {
        logger()->warning("آزمون مورد نظر یافت نشد یا به پایان رسیده است", [
            'chat_id' => $chat_id,
            'exam_id' => $examId
        ]);
        sendMessage($chat_id, "آزمون مورد نظر یافت نشد یا به پایان رسیده است.");
        return;
    }
    
    // دریافت کل سوالات آزمون
    $stmt = $pdo->prepare("SELECT * FROM questions WHERE exam_id = :exam_id ORDER BY id");
    $stmt->execute([':exam_id' => $examId]);
    $questions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (empty($questions)) {
        logger()->warning("آزمون هیچ سوالی ندارد", [
            'chat_id' => $chat_id,
            'exam_id' => $examId
        ]);
        sendMessage($chat_id, "این آزمون هیچ سوالی ندارد!");
        return;
    }
    
    // بررسی شماره سوال
    if ($questionNumber < 1 || $questionNumber > count($questions)) {
        logger()->warning("شماره سوال نامعتبر است", [
            'chat_id' => $chat_id,
            'exam_id' => $examId,
            'question_number' => $questionNumber,
            'total_questions' => count($questions)
        ]);
        sendMessage($chat_id, "شماره سوال نامعتبر است.");
        return;
    }
    
    // دریافت سوال فعلی
    $currentQuestion = $questions[$questionNumber - 1];

    // ثبت یا به‌روزرسانی زمان شروع سوال برای کاربر (در جدول user_question_sessions)
    // دریافت ID کاربر از chat_id
    $stmt = $pdo->prepare("SELECT id FROM users WHERE telegram_id = :telegram_id");
    $stmt->execute([':telegram_id' => $chat_id]);
    $userId = $stmt->fetchColumn();
    if ($userId) {
        $now = date('Y-m-d H:i:s');
        // درج یا به‌روزرسانی رکورد session
        $stmt = $pdo->prepare("INSERT INTO user_question_sessions (user_id, question_id, start_time) VALUES (:user_id, :question_id, :start_time) ON DUPLICATE KEY UPDATE start_time = :start_time");
        $stmt->execute([
            ':user_id' => $userId,
            ':question_id' => $currentQuestion['id'],
            ':start_time' => $now
        ]);
    }
    
    logger()->debug("اطلاعات سوال", [
        'question_id' => $currentQuestion['id'],
        'question_type' => $currentQuestion['question_type'],
        'score' => $currentQuestion['score']
    ]);
    
    // آماده‌سازی متن سوال
    $questionText = "*سوال {$questionNumber} از " . count($questions) . "*\n";
    // نمایش سطح دشواری اگر مقدار داشت
    if (!empty($currentQuestion['difficulty_level'])) {
        $diffLabel = '';
        switch($currentQuestion['difficulty_level']) {
            case 'easy': $diffLabel = 'آسان'; break;
            case 'medium': $diffLabel = 'متوسط'; break;
            case 'hard': $diffLabel = 'سخت'; break;
        }
        $questionText .= "[سطح: {$diffLabel}]\n";
    }
    // نمایش تصویر اگر مقدار داشت
    if (!empty($currentQuestion['image_url'])) {
        $questionText .= "[تصویر سوال](" . $currentQuestion['image_url'] . ")\n";
    }
    $questionText .= "\n{$currentQuestion['question_text']}\n\n";
    $questionText .= "امتیاز: {$currentQuestion['score']}";
    
    $inline_keyboard = [];
    
    // اگر سوال چندگزینه‌ای است، گزینه‌ها را نمایش بده
    if ($currentQuestion['question_type'] === 'multiple') {
        $stmt = $pdo->prepare("SELECT * FROM options WHERE question_id = :question_id ORDER BY id");
        $stmt->execute([':question_id' => $currentQuestion['id']]);
        $options = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        logger()->debug("گزینه‌های سوال چندگزینه‌ای", [
            'question_id' => $currentQuestion['id'],
            'options_count' => count($options)
        ]);
        
        foreach ($options as $option) {
            $optionText = '🔘 ' . $option['option_text'];
            // افزودن امتیاز گزینه اگر مقدار داشت
            if (isset($option['option_score']) && $option['option_score'] !== null) {
                $optionText .= " (امتیاز: {$option['option_score']})";
            }
            // افزودن لینک تصویر گزینه اگر مقدار داشت
            if (!empty($option['image_url'])) {
                $optionText .= " [تصویر](" . $option['image_url'] . ")";
            }
            $inline_keyboard[] = [
                ['text' => $optionText, 'callback_data' => 'exam_answer_' . $currentQuestion['id'] . '_' . $option['id']]
            ];
        }
        // دکمه رد کردن سوال
        $inline_keyboard[] = [
            ['text' => '⏭ رد کردن سوال', 'callback_data' => 'exam_skip_' . $currentQuestion['id']]
        ];
    } else {
        logger()->debug("سوال از نوع تشریحی است", [
            'question_id' => $currentQuestion['id']
        ]);
        
        // برای سوالات تشریحی، پیام راهنما نمایش داده شود
        $questionText .= "\n\n*این سوال تشریحی است. لطفاً پاسخ خود را به صورت یک پیام متنی ارسال کنید.*";
        
        // دکمه تایید پاسخ تشریحی
        $inline_keyboard[] = [
            ['text' => '✅ تایید پاسخ و ادامه', 'callback_data' => 'exam_answer_' . $currentQuestion['id'] . '_0']
        ];
    }
    
    // دکمه رد سوال (skip) برای همه سوالات
    $skipButton = [['text' => '⏭ رد سوال', 'callback_data' => 'exam_skip_' . $currentQuestion['id']]];
    $inline_keyboard[] = $skipButton;

    // دکمه‌های ناوبری
    $navButtons = [];
    
    // دکمه قبلی
    if ($questionNumber > 1) {
        $navButtons[] = ['text' => '◀️ قبلی', 'callback_data' => 'exam_question_' . $examId . '_' . ($questionNumber - 1)];
    }
    
    // دکمه بعدی یا پایان
    if ($questionNumber < count($questions)) {
        $navButtons[] = ['text' => 'بعدی ▶️', 'callback_data' => 'exam_question_' . $examId . '_' . ($questionNumber + 1)];
    } else {
        $navButtons[] = ['text' => '🏁 پایان آزمون', 'callback_data' => 'exam_finish_' . $examId];
    }
    
    $inline_keyboard[] = $navButtons;
    
    $reply_markup = ['inline_keyboard' => $inline_keyboard];
    sendMessage($chat_id, $questionText, $reply_markup, "Markdown");
    
    // دریافت ID کاربر
    $stmt = $pdo->prepare("SELECT id FROM users WHERE telegram_id = :telegram_id");
    $stmt->execute([':telegram_id' => $chat_id]);
    $userId = $stmt->fetchColumn();
    
    // ثبت لاگ مشاهده سوال
    $stmt = $pdo->prepare("
        INSERT INTO exam_logs (exam_id, user_id, action, details) 
        VALUES (:exam_id, :user_id, 'view_question', :details)
    ");
    $stmt->execute([
        ':exam_id' => $examId, 
        ':user_id' => $userId,
        ':details' => "مشاهده سوال {$questionNumber}"
    ]);
}

/**
 * ثبت پاسخ کاربر به سوال
 */
function recordAnswer($chat_id, $questionId, $optionId) {
    global $pdo;
    
    logger()->info("ثبت پاسخ کاربر به سوال", [
        'chat_id' => $chat_id,
        'question_id' => $questionId,
        'option_id' => $optionId,
        'function' => 'recordAnswer'
    ]);
    
    // دریافت اطلاعات سوال
    $stmt = $pdo->prepare("SELECT * FROM questions WHERE id = :id");
    $stmt->execute([':id' => $questionId]);
    $question = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$question) {
        sendMessage($chat_id, "سوال مورد نظر یافت نشد.");
        return;
    }
    
    // دریافت ID کاربر
    $stmt = $pdo->prepare("SELECT id FROM users WHERE telegram_id = :telegram_id");
    $stmt->execute([':telegram_id' => $chat_id]);
    $userId = $stmt->fetchColumn();
    
    // بررسی وجود نتیجه قبلی
    $stmt = $pdo->prepare("
        SELECT * FROM results 
        WHERE exam_id = :exam_id AND user_id = :user_id
    ");
    $stmt->execute([':exam_id' => $question['exam_id'], ':user_id' => $userId]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // اگر نتیجه‌ای وجود ندارد، ایجاد کن
    if (!$result) {
        $stmt = $pdo->prepare("INSERT INTO results (exam_id, user_id, score) VALUES (:exam_id, :user_id, 0)");
        $stmt->execute([':exam_id' => $question['exam_id'], ':user_id' => $userId]);
        $resultId = $pdo->lastInsertId();
    } else {
        $resultId = $result['id'];
    }
    
    // حذف پاسخ‌های قبلی کاربر به این سوال (اگر وجود داشته باشد)
    $stmt = $pdo->prepare("DELETE FROM user_answers WHERE question_id = :question_id AND user_id = :user_id");
    $stmt->execute([':question_id' => $questionId, ':user_id' => $userId]);
    
    // ثبت پاسخ جدید با منطق امتیازدهی و وضعیت
    if ($question['question_type'] === 'multiple' && $optionId > 0) {
        // دریافت اطلاعات گزینه انتخابی
        $stmt = $pdo->prepare("SELECT is_correct, option_score FROM options WHERE id = :id");
        $stmt->execute([':id' => $optionId]);
        $option = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $isCorrect = $option && $option['is_correct'] ? 1 : 0;
        $optionScore = isset($option['option_score']) ? floatval($option['option_score']) : 0;
        $negativeScore = isset($question['negative_score']) ? floatval($question['negative_score']) : 0;
        $answerStatus = $isCorrect ? 'correct' : 'wrong';
        $scoreChange = 0;
        
        if ($isCorrect) {
            $scoreChange = $optionScore;
        } else {
            $scoreChange = -$negativeScore;
        }
        // محاسبه مدت زمان پاسخ‌دهی (response_duration)
        $responseDuration = null;
        $stmt = $pdo->prepare("SELECT start_time FROM user_question_sessions WHERE user_id = :user_id AND question_id = :question_id");
        $stmt->execute([':user_id' => $userId, ':question_id' => $questionId]);
        $startTime = $stmt->fetchColumn();
        if ($startTime) {
            $responseDuration = strtotime(date('Y-m-d H:i:s')) - strtotime($startTime);
        }
        // ثبت پاسخ کاربر با امتیاز و وضعیت و مدت پاسخ‌دهی
        $stmt = $pdo->prepare(
            "INSERT INTO user_answers (question_id, user_id, option_id, option_score, answer_status, response_duration)
            VALUES (:question_id, :user_id, :option_id, :option_score, :answer_status, :response_duration)"
        );
        $stmt->execute([
            ':question_id' => $questionId,
            ':user_id' => $userId,
            ':option_id' => $optionId,
            ':option_score' => $scoreChange,
            ':answer_status' => $answerStatus,
            ':response_duration' => $responseDuration
        ]);
        // حذف رکورد session مربوط به این سوال
        $stmt = $pdo->prepare("DELETE FROM user_question_sessions WHERE user_id = :user_id AND question_id = :question_id");
        $stmt->execute([':user_id' => $userId, ':question_id' => $questionId]);
        // به‌روزرسانی نمره کاربر
        $stmt = $pdo->prepare("UPDATE results SET score = score + :score WHERE id = :id");
        $stmt->execute([':score' => $scoreChange, ':id' => $resultId]);
        
        $msg = $isCorrect ? "✅ پاسخ صحیح! امتیاز این گزینه: {$optionScore}" : "❌ پاسخ اشتباه. نمره منفی: {$negativeScore}";
        $msg .= "\nلطفاً به سوال بعدی پاسخ دهید یا روی 'پایان آزمون' کلیک کنید.";
        sendMessage($chat_id, $msg);
    } 
    elseif ($question['question_type'] === 'descriptive') {
        // محاسبه مدت زمان پاسخ‌دهی (response_duration)
        $responseDuration = null;
        $stmt = $pdo->prepare("SELECT start_time FROM user_question_sessions WHERE user_id = :user_id AND question_id = :question_id");
        $stmt->execute([':user_id' => $userId, ':question_id' => $questionId]);
        $startTime = $stmt->fetchColumn();
        if ($startTime) {
            $responseDuration = strtotime(date('Y-m-d H:i:s')) - strtotime($startTime);
        }
        // برای سوال تشریحی، فقط وضعیت ثبت می‌شود (بدون نمره)
        $stmt = $pdo->prepare(
            "INSERT INTO user_answers (question_id, user_id, answer_status, response_duration)
            VALUES (:question_id, :user_id, :answer_status, :response_duration)"
        );
        $stmt->execute([
            ':question_id' => $questionId,
            ':user_id' => $userId,
            ':answer_status' => 'descriptive',
            ':response_duration' => $responseDuration
        ]);
        // حذف رکورد session مربوط به این سوال
        $stmt = $pdo->prepare("DELETE FROM user_question_sessions WHERE user_id = :user_id AND question_id = :question_id");
        $stmt->execute([':user_id' => $userId, ':question_id' => $questionId]);
        sendMessage($chat_id, "✅ پاسخ شما به سوال تشریحی ثبت شد. لطفاً به سوال بعدی پاسخ دهید یا روی 'پایان آزمون' کلیک کنید.");
    }
    
    // ثبت لاگ پاسخ
    $stmt = $pdo->prepare("
        INSERT INTO exam_logs (exam_id, user_id, action, details) 
        VALUES (:exam_id, :user_id, 'answer', :details)
    ");
    $stmt->execute([
        ':exam_id' => $question['exam_id'], 
        ':user_id' => $userId,
        ':details' => "پاسخ به سوال {$questionId}"
    ]);
}

/**
 * پایان آزمون و نمایش نتیجه
 */
function finishExam($chat_id, $examId) {
    global $pdo;
    
    logger()->info("پایان آزمون و محاسبه نتیجه", [
        'chat_id' => $chat_id,
        'exam_id' => $examId,
        'function' => 'finishExam'
    ]);
    
    // دریافت اطلاعات آزمون
    $stmt = $pdo->prepare("SELECT * FROM exams WHERE id = :id");
    $stmt->execute([':id' => $examId]);
    $exam = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$exam) {
        sendMessage($chat_id, "آزمون مورد نظر یافت نشد.");
        return;
    }
    
    // دریافت ID کاربر
    $stmt = $pdo->prepare("SELECT id FROM users WHERE telegram_id = :telegram_id");
    $stmt->execute([':telegram_id' => $chat_id]);
    $userId = $stmt->fetchColumn();
    
    // دریافت نتیجه آزمون
    $stmt = $pdo->prepare("
        SELECT * FROM results 
        WHERE exam_id = :exam_id AND user_id = :user_id
    ");
    $stmt->execute([':exam_id' => $examId, ':user_id' => $userId]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // محاسبه امتیاز کل آزمون
    $stmt = $pdo->prepare("SELECT SUM(score) FROM questions WHERE exam_id = :exam_id");
    $stmt->execute([':exam_id' => $examId]);
    $totalScore = $stmt->fetchColumn();
    
    // نمایش نتیجه
    $resultText = "🏁 *پایان آزمون*\n\n";
    $resultText .= "*{$exam['title']}*\n\n";
    
    if ($result) {
        $resultText .= "📊 *نتیجه شما:*\n";
        $resultText .= "- نمره کسب شده: {$result['score']} از {$totalScore}\n";
        $resultText .= "- درصد: " . round(($result['score'] / $totalScore) * 100, 2) . "%\n";
        // --- آمار پاسخ‌ها ---
        $stmt = $pdo->prepare("SELECT answer_status, COUNT(*) as cnt, SUM(response_duration) as total_time FROM user_answers WHERE user_id = :user_id AND question_id IN (SELECT id FROM questions WHERE exam_id = :exam_id) GROUP BY answer_status");
        $stmt->execute([':user_id' => $userId, ':exam_id' => $examId]);
        $stats = $stmt->fetchAll(PDO::FETCH_KEY_PAIR | PDO::FETCH_GROUP);
        $correct = isset($stats['correct']) ? $stats['correct'][0]['cnt'] : 0;
        $wrong = isset($stats['wrong']) ? $stats['wrong'][0]['cnt'] : 0;
        $skipped = isset($stats['skipped']) ? $stats['skipped'][0]['cnt'] : 0;
        $desc = isset($stats['descriptive']) ? $stats['descriptive'][0]['cnt'] : 0;
        $totalQuestions = $pdo->query("SELECT COUNT(*) FROM questions WHERE exam_id = " . intval($examId))->fetchColumn();
        $answered = $correct + $wrong + $desc;
        $resultText .= "\n- تعداد صحیح: {$correct}";
        $resultText .= "\n- تعداد غلط: {$wrong}";
        $resultText .= "\n- تعداد رد شده: {$skipped}";
        $resultText .= "\n- تشریحی: {$desc}";
        $resultText .= "\n- مجموع سوالات: {$totalQuestions}";
        $resultText .= "\n- درصد صحیح: " . ($totalQuestions ? round($correct/$totalQuestions*100,1) : 0) . "%";
        $resultText .= "\n- درصد غلط: " . ($totalQuestions ? round($wrong/$totalQuestions*100,1) : 0) . "%";
        $resultText .= "\n- درصد رد شده: " . ($totalQuestions ? round($skipped/$totalQuestions*100,1) : 0) . "%";
        // --- زمان پاسخ‌دهی ---
        $stmt = $pdo->prepare("SELECT SUM(response_duration) as sum_time, AVG(response_duration) as avg_time FROM user_answers WHERE user_id = :user_id AND question_id IN (SELECT id FROM questions WHERE exam_id = :exam_id)");
        $stmt->execute([':user_id' => $userId, ':exam_id' => $examId]);
        $timing = $stmt->fetch(PDO::FETCH_ASSOC);
        $sumTime = $timing['sum_time'] ? intval($timing['sum_time']) : 0;
        $avgTime = $timing['avg_time'] ? round($timing['avg_time'],1) : 0;
        $resultText .= "\n- مجموع زمان پاسخ: {$sumTime} ثانیه";
        $resultText .= "\n- میانگین زمان هر سوال: {$avgTime} ثانیه";
        // --- رتبه کاربر ---
        $stmt = $pdo->prepare("SELECT user_id, score FROM results WHERE exam_id = :exam_id ORDER BY score DESC, id ASC");
        $stmt->execute([':exam_id' => $examId]);
        $ranks = $stmt->fetchAll(PDO::FETCH_ASSOC);
        $rank = 0;
        foreach ($ranks as $i => $row) {
            if ($row['user_id'] == $userId) {
                $rank = $i + 1;
                break;
            }
        }
        $resultText .= "\n- رتبه شما: {$rank} از " . count($ranks);
    } else {
        $resultText .= "⚠️ شما به هیچ سوالی پاسخ نداده‌اید!\n";
    }
    
    $resultText .= "\nبا تشکر از شرکت شما در این آزمون.";
    
    $backButton = [
        [['text' => '🔙 بازگشت به منوی اصلی', 'callback_data' => 'exam_back_to_menu']]
    ];
    
    $reply_markup = ['inline_keyboard' => $backButton];
    sendMessage($chat_id, $resultText, $reply_markup, "Markdown");
    
    // ثبت لاگ اتمام آزمون
    $stmt = $pdo->prepare("
        INSERT INTO exam_logs (exam_id, user_id, action, details) 
        VALUES (:exam_id, :user_id, 'finish', :details)
    ");
    $stmt->execute([
        ':exam_id' => $examId, 
        ':user_id' => $userId,
        ':details' => "اتمام آزمون با نمره " . ($result ? $result['score'] : '0')
    ]);
}

/**
 * نمایش پنل مدیریت آزمون (برای ادمین)
 */
function showExamAdminPanel($chat_id) {
    global $pdo;
    
    logger()->info("نمایش پنل مدیریت آزمون‌ها برای ادمین", [
        'chat_id' => $chat_id,
        'function' => 'showExamAdminPanel'
    ]);
    
    $inline_keyboard = [
        [
            ['text' => '🆕 ✨ افزودن آزمون جدید', 'callback_data' => 'admin_add_exam']
        ],
        [
            ['text' => '🗂️ مدیریت آزمون‌ها', 'callback_data' => 'admin_manage_exams']
        ],
        [
            ['text' => '📈 گزارش نتایج و رتبه‌ها', 'callback_data' => 'admin_exam_results']
        ],
        [
            ['text' => '🏠 بازگشت به داشبورد اصلی', 'callback_data' => 'admin_back_to_panel']
        ]
    ];
    
    $reply_markup = ['inline_keyboard' => $inline_keyboard];
    $adminPanelText = "<b>🛠️ پنل مدیریت آزمون‌ها</b>\n\n";
    $adminPanelText .= "به پنل مدیریت آزمون‌ها خوش آمدید! از گزینه‌های زیر برای مدیریت کامل آزمون‌ها استفاده کنید.\n";
    $adminPanelText .= "<i>هر بخش امکانات ویژه‌ای برای ساخت، ویرایش و مشاهده گزارش‌ها دارد.</i>\n";
    sendMessage($chat_id, $adminPanelText, $reply_markup, "HTML");
}

/**
 * تولید لینک آزمون وب برای کاربر
 */
function generateWebExamLink($chat_id, $exam_id) {
    global $pdo;
    
    logger()->info("تولید لینک آزمون وب", [
        'chat_id' => $chat_id,
        'exam_id' => $exam_id,
        'function' => 'generateWebExamLink'
    ]);
    
    // دریافت آیدی کاربر
    $stmt = $pdo->prepare("SELECT id FROM users WHERE telegram_id = :telegram_id");
    $stmt->execute([':telegram_id' => $chat_id]);
    $user_id = $stmt->fetchColumn();
    
    if (!$user_id) {
        logger()->warning("کاربر یافت نشد", ['chat_id' => $chat_id]);
        return false;
    }
    
    // ساخت توکن امنیتی
    $token = md5($user_id . $exam_id . 'SECRET_KEY_CHANGE_THIS');
    
    // تولید URL
    $base_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://" . $_SERVER['HTTP_HOST'];
    $exam_url = $base_url . "/exam.php?exam_id={$exam_id}&user_id={$user_id}&token={$token}";
    
    logger()->debug("لینک آزمون وب تولید شد", [
        'chat_id' => $chat_id,
        'exam_id' => $exam_id,
        'url' => $exam_url
    ]);
    
    return $exam_url;
}
/**
 * ثبت رد کردن سوال توسط کاربر (skipped)
 */
function skipQuestion($chat_id, $questionId) {
    global $pdo;
    
    logger()->info("رد کردن سوال توسط کاربر", [
        'chat_id' => $chat_id,
        'question_id' => $questionId,
        'function' => 'skipQuestion'
    ]);
    
    // دریافت اطلاعات سوال
    $stmt = $pdo->prepare("SELECT * FROM questions WHERE id = :id");
    $stmt->execute([':id' => $questionId]);
    $question = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$question) {
        sendMessage($chat_id, "سوال مورد نظر یافت نشد.");
        return;
    }
    // دریافت ID کاربر
    $stmt = $pdo->prepare("SELECT id FROM users WHERE telegram_id = :telegram_id");
    $stmt->execute([':telegram_id' => $chat_id]);
    $userId = $stmt->fetchColumn();
    if (!$userId) {
        sendMessage($chat_id, "کاربر یافت نشد.");
        return;
    }
    // حذف پاسخ قبلی کاربر به این سوال (اگر وجود داشته باشد)
    $stmt = $pdo->prepare("DELETE FROM user_answers WHERE question_id = :question_id AND user_id = :user_id");
    $stmt->execute([':question_id' => $questionId, ':user_id' => $userId]);
    // محاسبه مدت زمان پاسخ‌دهی (response_duration)
    $responseDuration = null;
    $stmt = $pdo->prepare("SELECT start_time FROM user_question_sessions WHERE user_id = :user_id AND question_id = :question_id");
    $stmt->execute([':user_id' => $userId, ':question_id' => $questionId]);
    $startTime = $stmt->fetchColumn();
    if ($startTime) {
        $responseDuration = strtotime(date('Y-m-d H:i:s')) - strtotime($startTime);
    }
    // ثبت پاسخ رد شده
    $stmt = $pdo->prepare(
        "INSERT INTO user_answers (question_id, user_id, answer_status, response_duration)
        VALUES (:question_id, :user_id, :answer_status, :response_duration)"
    );
    $stmt->execute([
        ':question_id' => $questionId,
        ':user_id' => $userId,
        ':answer_status' => 'skipped',
        ':response_duration' => $responseDuration
    ]);
    // حذف رکورد session مربوط به این سوال
    $stmt = $pdo->prepare("DELETE FROM user_question_sessions WHERE user_id = :user_id AND question_id = :question_id");
    $stmt->execute([':user_id' => $userId, ':question_id' => $questionId]);
    // پیام به کاربر
    sendMessage($chat_id, "⏭ سوال رد شد. به سوال بعدی بروید یا آزمون را به پایان برسانید.");
    // هدایت به سوال بعدی یا پایان آزمون
    // دریافت شماره و لیست سوالات آزمون
    $stmt = $pdo->prepare("SELECT exam_id FROM questions WHERE id = :id");
    $stmt->execute([':id' => $questionId]);
    $examId = $stmt->fetchColumn();
    $stmt = $pdo->prepare("SELECT id FROM questions WHERE exam_id = :exam_id ORDER BY id");
    $stmt->execute([':exam_id' => $examId]);
    $questions = $stmt->fetchAll(PDO::FETCH_COLUMN);
    $currentIndex = array_search($questionId, $questions);
    if ($currentIndex !== false && $currentIndex < count($questions) - 1) {
        // سوال بعدی
        $nextQuestionNumber = $currentIndex + 2; // شماره سوال بعدی (1-based)
        showExamQuestion($chat_id, $examId, $nextQuestionNumber);
    } else {
        // پایان آزمون
        finishExam($chat_id, $examId);
    }
}

?>
