<?php
// bot.php
require_once 'config.php';
require_once 'telegram_api.php';
require_once 'exam_functions.php';
require_once 'admin_functions.php';

// شروع لاگ‌گیری برای اجرای ربات
logger()->info("شروع اجرای ربات تلگرام", [
    'time' => date('Y-m-d H:i:s'),
    'script' => 'bot.php'
]);

// اتصال به پایگاه داده
try {
    $pdo = new PDO('mysql:host=' . DB_HOST . ';dbname=' . DB_NAME . ';charset=utf8mb4', DB_USER, DB_PASS);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->exec("SET NAMES utf8mb4");
    logger()->info("اتصال به پایگاه داده با موفقیت برقرار شد");
} catch (PDOException $e) {
    logger()->databaseError($e);
    exit;
}

// دریافت آپدیت از تلگرام
$content = file_get_contents("php://input");
$update = json_decode($content, true);
if (!$update) {
    logger()->warning("داده‌ی JSON نامعتبر یا خالی دریافت شد", ['content' => $content]);
    exit;
}

// ثبت لاگ برای داده‌های دریافتی از تلگرام
logger()->debug("داده‌های دریافتی از تلگرام", ['update' => $update]);

/* 
  اگر کاربر دستور /suggest ارسال کند، یک پیام دعوت به ارسال فایل پیشنهادی نمایش داده می‌شود.
  همچنین اگر پیام شامل یک فایل (document, video, audio) باشد و در کپشن آن "/suggest" نوشته شده باشد،
  آن فایل به عنوان فایل پیشنهادی ثبت می‌شود.
*/

if (isset($update['message'])) {
    $message = $update['message'];
    $chat_id = $message['chat']['id'];
    $text = isset($message['text']) ? trim($message['text']) : '';

    // ثبت لاگ پیام دریافتی از کاربر
    logger()->incomingMessage($message);
    
    // استخراج اطلاعات کاربر از قسمت 'from'
    $from = $message['from'];
    $username = isset($from['username']) ? $from['username'] : '';
    $full_name = trim(($from['first_name'] ?? '') . ' ' . ($from['last_name'] ?? ''));

    // مدیریت کاربران: ثبت یا به‌روزرسانی کاربر (با ذخیره username و full_name)
    $stmt = $pdo->prepare("SELECT * FROM users WHERE telegram_id = :telegram_id LIMIT 1");
    $stmt->execute([':telegram_id' => $chat_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($user) {
        if ($user['status'] == 'blocked') {
            logger()->info("درخواست از کاربر مسدود شده", [
                'chat_id' => $chat_id,
                'username' => $username,
                'full_name' => $full_name
            ]);
            sendMessage($chat_id, "متأسفانه شما مسدود شده‌اید.");
            exit;
        }
        // به‌روزرسانی زمان آخرین استفاده و اطلاعات کاربری
        $stmtUpdate = $pdo->prepare("UPDATE users SET last_usage = NOW(), username = :username, full_name = :full_name WHERE id = :id");
        $stmtUpdate->execute([':username' => $username, ':full_name' => $full_name, ':id' => $user['id']]);
        logger()->debug("به‌روزرسانی اطلاعات کاربر", [
            'user_id' => $user['id'],
            'chat_id' => $chat_id,
            'username' => $username,
            'full_name' => $full_name
        ]);
    } else {
        // ثبت کاربر جدید
        $stmtInsert = $pdo->prepare("INSERT INTO users (telegram_id, username, full_name) VALUES (:telegram_id, :username, :full_name)");
        $stmtInsert->execute([':telegram_id' => $chat_id, ':username' => $username, ':full_name' => $full_name]);
        $newUserId = $pdo->lastInsertId();
        logger()->info("کاربر جدید ثبت شد", [
            'user_id' => $newUserId,
            'chat_id' => $chat_id,
            'username' => $username,
            'full_name' => $full_name
        ]);
    }

    // بررسی دسترسی ادمین
    $isAdmin = false;
    $adminIds = explode(',', ADMIN_IDS);
    if (in_array($chat_id, $adminIds)) {
        $isAdmin = true;
        logger()->debug("کاربر دارای دسترسی ادمین است", ['chat_id' => $chat_id]);
    }

    // نمایش پنل مدیریت برای ادمین‌ها با دستور /admin
    if ($isAdmin && $text === '/admin') {
        logger()->info("نمایش پنل مدیریت برای ادمین", ['chat_id' => $chat_id]);
        // ایجاد دکمه‌های کیبوردی برای پنل مدیریت
        $keyboard = [
            ['📊 داشبورد و آمار'],
            ['➕ افزودن دکمه اصلی', '➕ افزودن دکمه اینلاین'],
            ['📝 مدیریت دکمه‌های اصلی', '📝 مدیریت دکمه‌های اینلاین'],
            ['📂 آپلود فایل', '📝 مدیریت فایل‌ها'],
            ['👥 مدیریت کاربران', '📨 فایل‌های پیشنهادی'],
            ['📋 مدیریت آزمون‌ها'],
            ['🔙 بازگشت به منوی اصلی']
        ];
        
        $reply_markup = [
            'keyboard' => $keyboard,
            'resize_keyboard' => true,
            'one_time_keyboard' => false
        ];
        
        sendMessage($chat_id, "✅ به پنل مدیریت ربات خوش آمدید.\n\nلطفاً یکی از گزینه‌های زیر را انتخاب کنید:", $reply_markup);
        exit;
    }
    
    // پردازش انتخاب‌های پنل مدیریت (از دکمه‌های کیبوردی)
    if ($isAdmin) {
        // پردازش دکمه‌های کیبوردی مدیریتی
        switch ($text) {
            case '📊 داشبورد و آمار':
                logger()->info("درخواست نمایش داشبورد آمار", ['chat_id' => $chat_id]);
                // نمایش داشبورد آماری
                $totalUsers = $pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();
                $activeUsers = $pdo->query("SELECT COUNT(*) FROM users WHERE status = 'active'")->fetchColumn();
                $blockedUsers = $pdo->query("SELECT COUNT(*) FROM users WHERE status = 'blocked'")->fetchColumn();
                $totalMainButtons = $pdo->query("SELECT COUNT(*) FROM main_buttons")->fetchColumn();
                $totalInlineButtons = $pdo->query("SELECT COUNT(*) FROM inline_buttons")->fetchColumn();
                $totalFiles = $pdo->query("SELECT COUNT(*) FROM files")->fetchColumn();
                $pendingSuggestions = $pdo->query("SELECT COUNT(*) FROM suggested_files WHERE status = 'pending'")->fetchColumn();
                $totalExams = $pdo->query("SELECT COUNT(*) FROM exams")->fetchColumn();
                $activeExams = $pdo->query("SELECT COUNT(*) FROM exams WHERE start_time <= NOW() AND end_time >= NOW()")->fetchColumn();
                
                $stats = "📊 *داشبورد مدیریت*\n\n".
                         "👥 *کاربران:*\n".
                         "- کل کاربران: $totalUsers\n".
                         "- کاربران فعال: $activeUsers\n".
                         "- کاربران مسدود: $blockedUsers\n\n".
                         "🔘 *دکمه‌ها:*\n".
                         "- دکمه‌های اصلی: $totalMainButtons\n".
                         "- دکمه‌های اینلاین: $totalInlineButtons\n\n".
                         "📂 *فایل‌ها:*\n".
                         "- کل فایل‌ها: $totalFiles\n".
                         "- فایل‌های پیشنهادی در انتظار: $pendingSuggestions\n\n".
                         "📝 *آزمون‌ها:*\n".
                         "- کل آزمون‌ها: $totalExams\n".
                         "- آزمون‌های فعال: $activeExams\n";
                
                sendMessage($chat_id, $stats, null, "Markdown");
                break;
                
            case '➕ افزودن دکمه اصلی':
                // فرم افزودن دکمه اصلی
                $formText = "برای افزودن دکمه اصلی، متن دکمه را با فرمت زیر ارسال کنید:\n\n".
                             "/add_main_button عنوان دکمه";
                
                sendMessage($chat_id, $formText);
                break;
                
            case '➕ افزودن دکمه اینلاین':
                // نمایش لیست دکمه‌های اصلی برای انتخاب
                $stmt = $pdo->query("SELECT id, title FROM main_buttons ORDER BY title");
                $mainButtons = [];
                
                while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    $mainButtons[] = [
                        ['text' => $row['title'], 'callback_data' => 'admin_select_main_' . $row['id']]
                    ];
                }
                
                $reply_markup = ['inline_keyboard' => $mainButtons];
                sendMessage($chat_id, "لطفاً دکمه اصلی مورد نظر برای افزودن دکمه اینلاین را انتخاب کنید:", $reply_markup);
                break;
                
            case '📝 مدیریت دکمه‌های اصلی':
                // نمایش لیست دکمه‌های اصلی برای مدیریت
                $stmt = $pdo->query("SELECT id, title FROM main_buttons ORDER BY title");
                $mainButtons = [];
                
                while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    $mainButtons[] = [
                        ['text' => $row['title'], 'callback_data' => 'admin_edit_main_' . $row['id']],
                        ['text' => '❌', 'callback_data' => 'admin_delete_main_' . $row['id']]
                    ];
                }
                
                $reply_markup = ['inline_keyboard' => $mainButtons];
                sendMessage($chat_id, "مدیریت دکمه‌های اصلی:\n\nبرای ویرایش روی عنوان دکمه، و برای حذف روی ❌ کلیک کنید.", $reply_markup);
                break;
                
            case '📝 مدیریت دکمه‌های اینلاین':
                // نمایش لیست دکمه‌های اینلاین برای مدیریت
                $stmt = $pdo->query("SELECT ib.id, ib.title, mb.title as main_title 
                                    FROM inline_buttons ib 
                                    JOIN main_buttons mb ON ib.main_button_id = mb.id 
                                    ORDER BY mb.title, ib.title");
                $inlineButtons = [];
                $currentMainTitle = '';
                
                while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    if ($currentMainTitle != $row['main_title']) {
                        $currentMainTitle = $row['main_title'];
                        $inlineButtons[] = [['text' => "📌 {$row['main_title']}", 'callback_data' => 'admin_no_action']];
                    }
                    
                    $inlineButtons[] = [
                        ['text' => $row['title'], 'callback_data' => 'admin_edit_inline_' . $row['id']],
                        ['text' => '❌', 'callback_data' => 'admin_delete_inline_' . $row['id']]
                    ];
                }
                
                $reply_markup = ['inline_keyboard' => $inlineButtons];
                sendMessage($chat_id, "مدیریت دکمه‌های اینلاین:\n\nبرای ویرایش روی عنوان دکمه، و برای حذف روی ❌ کلیک کنید.", $reply_markup);
                break;
                
            case '📂 آپلود فایل':
                // نمایش لیست دکمه‌های اینلاین برای انتخاب
                $stmt = $pdo->query("SELECT ib.id, ib.title, mb.title as main_title 
                                    FROM inline_buttons ib 
                                    JOIN main_buttons mb ON ib.main_button_id = mb.id 
                                    ORDER BY mb.title, ib.title");
                $inlineButtons = [];
                $currentMainTitle = '';
                
                while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    if ($currentMainTitle != $row['main_title']) {
                        $currentMainTitle = $row['main_title'];
                        $inlineButtons[] = [['text' => "📌 {$row['main_title']}", 'callback_data' => 'admin_no_action']];
                    }
                    
                    $inlineButtons[] = [['text' => $row['title'], 'callback_data' => 'admin_select_inline_' . $row['id']]];
                }
                
                $reply_markup = ['inline_keyboard' => $inlineButtons];
                sendMessage($chat_id, "لطفاً دکمه اینلاین مورد نظر برای آپلود فایل را انتخاب کنید:", $reply_markup);
                break;
                
            case '📝 مدیریت فایل‌ها':
                // نمایش لیست فایل‌ها برای مدیریت
                $stmt = $pdo->query("SELECT f.id, f.file_type, f.caption, ib.title as inline_title 
                                    FROM files f 
                                    JOIN inline_buttons ib ON f.inline_button_id = ib.id 
                                    ORDER BY ib.title, f.id");
                $fileButtons = [];
                $currentInlineTitle = '';
                
                while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    if ($currentInlineTitle != $row['inline_title']) {
                        $currentInlineTitle = $row['inline_title'];
                        $fileButtons[] = [['text' => "📌 {$row['inline_title']}", 'callback_data' => 'admin_no_action']];
                    }
                    
                    $caption = !empty($row['caption']) ? $row['caption'] : 'بدون عنوان';
                    $fileButtons[] = [
                        ['text' => "{$row['file_type']}: {$caption}", 'callback_data' => 'admin_view_file_' . $row['id']],
                        ['text' => '❌', 'callback_data' => 'admin_delete_file_' . $row['id']]
                    ];
                }
                
                $reply_markup = ['inline_keyboard' => $fileButtons];
                sendMessage($chat_id, "مدیریت فایل‌ها:\n\nبرای مشاهده فایل روی عنوان آن، و برای حذف روی ❌ کلیک کنید.", $reply_markup);
                break;
                
            case '👥 مدیریت کاربران':
                // نمایش گزینه‌های مدیریت کاربران
                $userOptions = [
                    [['text' => '📊 آمار کاربران', 'callback_data' => 'admin_user_stats']],
                    [['text' => '📑 لیست کاربران فعال', 'callback_data' => 'admin_list_active_users']],
                    [['text' => '🚫 لیست کاربران مسدود', 'callback_data' => 'admin_list_blocked_users']]
                ];
                
                $reply_markup = ['inline_keyboard' => $userOptions];
                sendMessage($chat_id, "مدیریت کاربران:\n\nلطفاً یکی از گزینه‌های زیر را انتخاب کنید:", $reply_markup);
                break;
                
            case '📨 فایل‌های پیشنهادی':
                // نمایش لیست فایل‌های پیشنهادی
                $stmt = $pdo->prepare("SELECT sf.id, sf.file_url, sf.status, u.telegram_id, u.username, u.full_name 
                                      FROM suggested_files sf 
                                      JOIN users u ON sf.user_id = u.id 
                                      WHERE sf.status = 'pending' 
                                      ORDER BY sf.id DESC
                                      LIMIT 10");
                $stmt->execute();
                $suggestions = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if (empty($suggestions)) {
                    sendMessage($chat_id, "در حال حاضر هیچ فایل پیشنهادی در انتظار بررسی وجود ندارد.");
                } else {
                    $suggestionButtons = [];
                    foreach ($suggestions as $suggestion) {
                        $username = !empty($suggestion['username']) ? "@{$suggestion['username']}" : $suggestion['full_name'];
                        $suggestionButtons[] = [
                            ['text' => "مشاهده فایل از: {$username}", 'callback_data' => 'admin_view_suggestion_' . $suggestion['id']]
                        ];
                        $suggestionButtons[] = [
                            ['text' => '✅ تایید', 'callback_data' => 'admin_approve_suggestion_' . $suggestion['id']],
                            ['text' => '❌ رد', 'callback_data' => 'admin_reject_suggestion_' . $suggestion['id']]
                        ];
                    }
                    
                    $reply_markup = ['inline_keyboard' => $suggestionButtons];
                    sendMessage($chat_id, "فایل‌های پیشنهادی در انتظار بررسی:", $reply_markup);
                }
                break;
                
            case '📋 مدیریت آزمون‌ها':
                // نمایش منوی مدیریت آزمون‌ها با دکمه‌های اینلاین بهبود یافته
                showAdminExamPanel($chat_id);
                break;
                
            case '🔙 بازگشت به منوی اصلی':
                // بازگشت به منوی اصلی
                $welcomeText = "سلام و خوش آمدید به ربات آموزشی ما.\n\n" .
                               "لطفاً از منوی زیر یکی از دسته‌های آموزشی را انتخاب نمایید.";
                
                // واکشی دکمه‌های اصلی از پایگاه داده
                $stmt = $pdo->query("SELECT * FROM main_buttons ORDER BY title");
                $keyboardButtons = [];
                while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                    $keyboardButtons[] = $row['title'];
                }
                
                // تقسیم دکمه‌ها به ردیف‌های چندتایی (مثلاً 3 در هر ردیف)
                $keyboard = [];
                $row = [];
                $i = 0;
                foreach ($keyboardButtons as $btn) {
                    $row[] = $btn;
                    $i++;
                    if ($i % 3 == 0) {
                        $keyboard[] = $row;
                        $row = [];
                    }
                }
                if (!empty($row)) {
                    $keyboard[] = $row;
                }
                
                // افزودن دکمه آزمون
                $keyboard[] = ['/exam'];
                
                // افزودن دکمه مدیریت برای ادمین‌ها
                if ($isAdmin) {
                    $keyboard[] = ['مدیریت ربات'];
                }
                
                $reply_markup = [
                    'keyboard' => $keyboard,
                    'resize_keyboard' => true,
                    'one_time_keyboard' => false
                ];
                sendMessage($chat_id, $welcomeText, $reply_markup);
                break;
        }
    }

    // پردازش دستورات مدیریتی (فقط برای ادمین‌ها)
    if ($isAdmin) {
        // افزودن دکمه اصلی
        if (strpos($text, '/add_main_button') === 0) {
            $title = trim(str_replace('/add_main_button', '', $text));
            if (!empty($title)) {
                $stmt = $pdo->prepare("INSERT INTO main_buttons (title) VALUES (:title)");
                if ($stmt->execute([':title' => $title])) {
                    sendMessage($chat_id, "✅ دکمه اصلی با عنوان \"$title\" با موفقیت اضافه شد.");
                } else {
                    sendMessage($chat_id, "❌ خطا در افزودن دکمه اصلی.");
                }
            } else {
                sendMessage($chat_id, "❌ عنوان دکمه اصلی نمی‌تواند خالی باشد.");
            }
            exit;
        }
        
        // افزودن دکمه اینلاین
        if (strpos($text, '/add_inline_button') === 0) {
            $params = trim(str_replace('/add_inline_button', '', $text));
            $parts = explode(' ', $params, 2);
            
            if (count($parts) >= 2) {
                $main_button_id = intval(trim($parts[0]));
                $titleParts = explode('|', $parts[1], 2);
                
                $inline_title = trim($titleParts[0]);
                $description = (count($titleParts) > 1) ? trim($titleParts[1]) : '';
                
                if ($main_button_id > 0 && !empty($inline_title)) {
                    $stmt = $pdo->prepare("INSERT INTO inline_buttons (main_button_id, title, description) VALUES (:mbid, :title, :description)");
                    if ($stmt->execute([':mbid' => $main_button_id, ':title' => $inline_title, ':description' => $description])) {
                        sendMessage($chat_id, "✅ دکمه اینلاین با عنوان \"$inline_title\" با موفقیت اضافه شد.");
                    } else {
                        sendMessage($chat_id, "❌ خطا در افزودن دکمه اینلاین.");
                    }
                } else {
                    sendMessage($chat_id, "❌ اطلاعات وارد شده نامعتبر است.");
                }
            } else {
                sendMessage($chat_id, "❌ فرمت دستور نادرست است. صحیح: /add_inline_button {شناسه دکمه اصلی} {عنوان} | {توضیحات}");
            }
            exit;
        }
        
        // افزودن آزمون جدید
        if (strpos($text, '/add_exam') === 0) {
            $params = trim(str_replace('/add_exam', '', $text));
            $parts = explode('|', $params);
            
            if (count($parts) >= 3) {
                $title = trim($parts[0]);
                $description = trim($parts[1]);
                $end_time = trim($parts[2]); // فرمت YYYY-MM-DD HH:MM:SS
                
                if (!empty($title) && !empty($end_time)) {
                    $stmt = $pdo->prepare("INSERT INTO exams (title, description, start_time, end_time) VALUES (:title, :description, NOW(), :end_time)");
                    if ($stmt->execute([':title' => $title, ':description' => $description, ':end_time' => $end_time])) {
                        $examId = $pdo->lastInsertId();
                        sendMessage($chat_id, "✅ آزمون جدید با عنوان \"$title\" با موفقیت ایجاد شد.\n\nشناسه آزمون: $examId\n\nبرای افزودن سوال از دستور زیر استفاده کنید:\n/add_question {شناسه آزمون} {متن سوال} | {نوع سوال: multiple یا descriptive} | {امتیاز}");
                    } else {
                        sendMessage($chat_id, "❌ خطا در ایجاد آزمون.");
                    }
                } else {
                    sendMessage($chat_id, "❌ اطلاعات وارد شده نامعتبر است.");
                }
            } else {
                sendMessage($chat_id, "❌ فرمت دستور نادرست است. صحیح: /add_exam {عنوان} | {توضیحات} | {زمان پایان}");
            }
            exit;
        }
        
        // افزودن سوال به آزمون
        if (strpos($text, '/add_question') === 0) {
            $params = trim(str_replace('/add_question', '', $text));
            $parts = explode(' ', $params, 2);
            
            if (count($parts) >= 2) {
                $exam_id = intval(trim($parts[0]));
                $questionData = explode('|', $parts[1]);
                
                if (count($questionData) >= 3 && $exam_id > 0) {
                    $question_text = trim($questionData[0]);
                    $question_type = trim($questionData[1]);
                    $score = intval(trim($questionData[2]));
                    
                    if (!empty($question_text) && in_array($question_type, ['multiple', 'descriptive']) && $score > 0) {
                        $stmt = $pdo->prepare("INSERT INTO questions (exam_id, question_text, question_type, score) VALUES (:exam_id, :question_text, :question_type, :score)");
                        if ($stmt->execute([':exam_id' => $exam_id, ':question_text' => $question_text, ':question_type' => $question_type, ':score' => $score])) {
                            $questionId = $pdo->lastInsertId();
                            
                            if ($question_type === 'multiple') {
                                $message = "✅ سوال چندگزینه‌ای با موفقیت اضافه شد.\n\nشناسه سوال: $questionId\n\nبرای افزودن گزینه از دستور زیر استفاده کنید:\n/add_option {شناسه سوال} {متن گزینه} | {صحیح: 0 یا 1}";
                            } else {
                                $message = "✅ سوال تشریحی با موفقیت اضافه شد.\n\nشناسه سوال: $questionId";
                            }
                            
                            sendMessage($chat_id, $message);
                        } else {
                            sendMessage($chat_id, "❌ خطا در افزودن سوال.");
                        }
                    } else {
                        sendMessage($chat_id, "❌ اطلاعات وارد شده نامعتبر است.");
                    }
                } else {
                    sendMessage($chat_id, "❌ فرمت دستور نادرست است.");
                }
            } else {
                sendMessage($chat_id, "❌ فرمت دستور نادرست است. صحیح: /add_question {شناسه آزمون} {متن سوال} | {نوع سوال} | {امتیاز}");
            }
            exit;
        }
        
        // افزودن گزینه به سوال چندگزینه‌ای
        if (strpos($text, '/add_option') === 0) {
            $params = trim(str_replace('/add_option', '', $text));
            $parts = explode(' ', $params, 2);
            
            if (count($parts) >= 2) {
                $question_id = intval(trim($parts[0]));
                $optionData = explode('|', $parts[1]);
                
                if (count($optionData) >= 2 && $question_id > 0) {
                    $option_text = trim($optionData[0]);
                    $is_correct = (trim($optionData[1]) === '1') ? 1 : 0;
                    
                    if (!empty($option_text)) {
                        // بررسی نوع سوال
                        $stmt = $pdo->prepare("SELECT question_type FROM questions WHERE id = :id");
                        $stmt->execute([':id' => $question_id]);
                        $questionType = $stmt->fetchColumn();
                        
                        if ($questionType === 'multiple') {
                            $stmt = $pdo->prepare("INSERT INTO options (question_id, option_text, is_correct) VALUES (:question_id, :option_text, :is_correct)");
                            if ($stmt->execute([':question_id' => $question_id, ':option_text' => $option_text, ':is_correct' => $is_correct])) {
                                sendMessage($chat_id, "✅ گزینه با موفقیت به سوال اضافه شد.");
                            } else {
                                sendMessage($chat_id, "❌ خطا در افزودن گزینه.");
                            }
                        } else {
                            sendMessage($chat_id, "❌ این سوال از نوع چندگزینه‌ای نیست.");
                        }
                    } else {
                        sendMessage($chat_id, "❌ متن گزینه نمی‌تواند خالی باشد.");
                    }
                } else {
                    sendMessage($chat_id, "❌ فرمت دستور نادرست است.");
                }
            } else {
                sendMessage($chat_id, "❌ فرمت دستور نادرست است. صحیح: /add_option {شناسه سوال} {متن گزینه} | {صحیح: 0 یا 1}");
            }
            exit;
        }
        
        // مدیریت کاربران
        if (strpos($text, '/block_user') === 0) {
            $telegram_id = trim(str_replace('/block_user', '', $text));
            if (!empty($telegram_id)) {
                $stmt = $pdo->prepare("UPDATE users SET status = 'blocked' WHERE telegram_id = :telegram_id");
                if ($stmt->execute([':telegram_id' => $telegram_id])) {
                    sendMessage($chat_id, "✅ کاربر با شناسه \"$telegram_id\" مسدود شد.");
                } else {
                    sendMessage($chat_id, "❌ خطا در مسدود کردن کاربر.");
                }
            } else {
                sendMessage($chat_id, "❌ شناسه کاربر نمی‌تواند خالی باشد.");
            }
            exit;
        }
        
        if (strpos($text, '/unblock_user') === 0) {
            $telegram_id = trim(str_replace('/unblock_user', '', $text));
            if (!empty($telegram_id)) {
                $stmt = $pdo->prepare("UPDATE users SET status = 'active' WHERE telegram_id = :telegram_id");
                if ($stmt->execute([':telegram_id' => $telegram_id])) {
                    sendMessage($chat_id, "✅ کاربر با شناسه \"$telegram_id\" فعال شد.");
                } else {
                    sendMessage($chat_id, "❌ خطا در فعال کردن کاربر.");
                }
            } else {
                sendMessage($chat_id, "❌ شناسه کاربر نمی‌تواند خالی باشد.");
            }
            exit;
        }
    }

    // بررسی دستور مدیریت (admin)
    if ($text === '/admin') {
        // بررسی دسترسی ادمین
        if ($isAdmin) {
            // نمایش پنل مدیریت
            showAdminPanel($chat_id);
        } else {
            sendMessage($chat_id, "شما دسترسی به پنل مدیریت را ندارید.");
        }
        exit;
    }
    // اگر دستور /suggest ارسال شد، پیام دعوت به ارسال فایل پیشنهادی نمایش داده شود.
    elseif ($text === '/suggest') {
        logger()->info("کاربر درخواست ارسال فایل پیشنهادی کرده است", [
            'chat_id' => $chat_id,
            'username' => $username
        ]);
        sendMessage($chat_id, "لطفاً فایل پیشنهادی خود را ارسال نمایید. در کپشن فایل، دستور /suggest را درج کنید.");
        exit;
    }
    // اگر دستور /exam ارسال شده باشد، منوی آزمون‌ها نمایش داده شود
    elseif ($text === '/exam') {
        logger()->info("کاربر درخواست مشاهده منوی آزمون‌ها کرده است", [
            'chat_id' => $chat_id,
            'username' => $username
        ]);
        showExamMenu($chat_id);
        exit;
    }
    
    // اگر پیام شامل فایل پیشنهادی باشد (و در کپشن آن "/suggest" نوشته شده باشد)
    if ((isset($message['document']) || isset($message['video']) || isset($message['audio'])) && isset($message['caption'])) {
        if (strpos($message['caption'], '/suggest') !== false) {
            logger()->info("کاربر یک فایل پیشنهادی ارسال کرده است", [
                'chat_id' => $chat_id,
                'username' => $username,
                'message' => $message
            ]);
            
            // ثبت فایل پیشنهادی در جدول suggested_files
            // برای مثال، برای سند، فایل_id را ذخیره می‌کنیم؛ برای سایر موارد مشابه عمل کنید.
            $file_id = "";
            $file_type = "";
            if (isset($message['document'])) {
                $file_id = $message['document']['file_id'];
                $file_type = "document";
            } elseif (isset($message['video'])) {
                $file_id = $message['video']['file_id'];
                $file_type = "video";
            } elseif (isset($message['audio'])) {
                $file_id = $message['audio']['file_id'];
                $file_type = "audio";
            }
            
            logger()->debug("اطلاعات فایل پیشنهادی", [
                'file_id' => $file_id,
                'file_type' => $file_type
            ]);
            
            // اطمینان از وجود کاربر (قبلاً ثبت شده است)
            $stmt = $pdo->prepare("SELECT id FROM users WHERE telegram_id = :telegram_id LIMIT 1");
            $stmt->execute([':telegram_id' => $chat_id]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            $user_id = $user ? $user['id'] : $pdo->lastInsertId();
            $stmt = $pdo->prepare("INSERT INTO suggested_files (user_id, file_url) VALUES (:user_id, :file_url)");
            if ($stmt->execute([':user_id' => $user_id, ':file_url' => $file_id])) {
                logger()->info("فایل پیشنهادی با موفقیت ثبت شد", [
                    'user_id' => $user_id,
                    'file_id' => $file_id
                ]);
                sendMessage($chat_id, "فایل پیشنهادی شما با موفقیت ثبت شد.");
            } else {
                logger()->error("خطا در ثبت فایل پیشنهادی", [
                    'user_id' => $user_id,
                    'file_id' => $file_id,
                    'error' => $pdo->errorInfo()
                ]);
                sendMessage($chat_id, "خطا در ثبت فایل پیشنهادی. لطفاً مجدداً تلاش کنید.");
            }
            exit;
        }
    }

    // پردازش سایر پیام‌ها:
    if ($text === '/start') {
        logger()->info("کاربر دستور /start را ارسال کرده است", [
            'chat_id' => $chat_id,
            'username' => $username
        ]);
        
        $welcomeText = "سلام و خوش آمدید به ربات آموزشی ما.\n\n" .
                       "لطفاً از منوی زیر یکی از دسته‌های آموزشی را انتخاب نمایید.";
        
        // واکشی دکمه‌های اصلی از پایگاه داده
        $stmt = $pdo->query("SELECT * FROM main_buttons ORDER BY title");
        $keyboardButtons = [];
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $keyboardButtons[] = $row['title'];
        }
        
        logger()->debug("دکمه‌های اصلی بازیابی شدند", [
            'count' => count($keyboardButtons)
        ]);
        
        // تقسیم دکمه‌ها به ردیف‌های چندتایی (مثلاً 3 در هر ردیف)
        $keyboard = [];
        $row = [];
        $i = 0;
        foreach ($keyboardButtons as $btn) {
            $row[] = $btn;
            $i++;
            if ($i % 3 == 0) {
                $keyboard[] = $row;
                $row = [];
            }
        }
        if (!empty($row)) {
            $keyboard[] = $row;
        }
        
        // افزودن دکمه آزمون
        $keyboard[] = ['/exam'];
        
        // افزودن دکمه مدیریت برای ادمین‌ها
        if ($isAdmin) {
            $keyboard[] = ['مدیریت ربات'];
        }
        
        $reply_markup = [
            'keyboard' => $keyboard,
            'resize_keyboard' => true,
            'one_time_keyboard' => false
        ];
        sendMessage($chat_id, $welcomeText, $reply_markup);
    } else {
        // فرض می‌کنیم متن دریافتی یکی از عناوین دکمه‌های اصلی است.
        if ($text === 'مدیریت ربات' && $isAdmin) {
            logger()->info("کاربر ادمین درخواست مدیریت ربات کرده است", [
                'chat_id' => $chat_id,
                'username' => $username
            ]);
            
            // استفاده از تابع showAdminPanel برای نمایش پنل مدیریت با دکمه‌های اینلاین
            showAdminPanel($chat_id);
            exit;
        }
        
        // بررسی دکمه‌های دیگر پنل مدیریت
        if ($isAdmin) {
            switch ($text) {
                case '📊 داشبورد و آمار':
                case '➕ افزودن دکمه اصلی':
                case '➕ افزودن دکمه اینلاین':
                case '📂 آپلود فایل':
                case '📝 مدیریت دکمه‌های اصلی':
                case '📝 مدیریت دکمه‌های اینلاین':
                case '📝 مدیریت فایل‌ها':
                case '👥 مدیریت کاربران':
                case '📨 فایل‌های پیشنهادی':
                case '📋 مدیریت آزمون‌ها':
                    // همه دکمه‌های پنل مدیریت را به حالت پنل ادمین اجرا کن
                    logger()->info("کاربر ادمین یکی از دکمه‌های پنل مدیریت را انتخاب کرده است", [
                        'chat_id' => $chat_id,
                        'button' => $text
                    ]);
                    showAdminPanel($chat_id);
                    exit;
            }
        }
        
        // بررسی دکمه‌های منوی آزمون
        if ($text === '/exam') {
            logger()->info("کاربر درخواست مشاهده منوی آزمون‌ها کرده است", [
                'chat_id' => $chat_id,
                'username' => $username
            ]);
            showExamMenu($chat_id);
            exit;
        }
        
        $stmt = $pdo->prepare("SELECT id FROM main_buttons WHERE title = :title LIMIT 1");
        $stmt->execute([':title' => $text]);
        $main = $stmt->fetch(PDO::FETCH_ASSOC);
        if ($main) {
            logger()->info("کاربر دکمه اصلی را انتخاب کرده است", [
                'chat_id' => $chat_id,
                'button_title' => $text,
                'button_id' => $main['id']
            ]);
            
            // واکشی دکمه‌های اینلاین مرتبط با دکمه اصلی انتخاب‌شده
            $stmt2 = $pdo->prepare("SELECT id, title FROM inline_buttons WHERE main_button_id = :mid ORDER BY title");
            $stmt2->execute([':mid' => $main['id']]);
            $inline_buttons = [];
            while ($row = $stmt2->fetch(PDO::FETCH_ASSOC)) {
                $inline_buttons[] = [
                    'text' => $row['title'],
                    'callback_data' => "ib_" . $row['id']
                ];
            }
            
            logger()->debug("دکمه‌های اینلاین برای دکمه اصلی بازیابی شدند", [
                'main_button_id' => $main['id'],
                'inline_buttons_count' => count($inline_buttons)
            ]);
            
            if (!empty($inline_buttons)) {
                // تقسیم دکمه‌ها به ردیف‌های 3تایی
                $inline_keyboard = [];
                $temp = [];
                $i = 0;
                foreach ($inline_buttons as $btn) {
                    $temp[] = $btn;
                    $i++;
                    if ($i % 3 == 0) {
                        $inline_keyboard[] = $temp;
                        $temp = [];
                    }
                }
                if (!empty($temp)) {
                    $inline_keyboard[] = $temp;
                }
                $reply_markup = ['inline_keyboard' => $inline_keyboard];
                $msg = "📚 لطفاً یک گزینه را از زیر منو انتخاب کنید:";
                sendMessage($chat_id, $msg, $reply_markup);
            } else {
                logger()->warning("هیچ دکمه اینلاینی برای دکمه اصلی یافت نشد", [
                    'main_button_id' => $main['id'],
                    'main_button_title' => $text
                ]);
                sendMessage($chat_id, "برای این دسته آموزشی زیر منویی تعریف نشده است.");
            }
        } else {
            logger()->warning("کاربر پیام نامعتبر ارسال کرده است", [
                'chat_id' => $chat_id,
                'text' => $text
            ]);
            sendMessage($chat_id, "لطفاً از منوی ارائه‌شده استفاده کنید.");
        }
    }
}

// پردازش callback_query (دکمه‌های اینلاین)
elseif (isset($update['callback_query'])) {
    $callback = $update['callback_query'];
    $data = $callback['data'];
    $chat_id = $callback['message']['chat']['id'];
    
    // ثبت لاگ کال‌بک دریافتی
    logger()->info("کال‌بک دریافت شد", [
        'chat_id' => $chat_id,
        'callback_data' => $data
    ]);
    
    // پردازش کال‌بک‌های مختلف
    if (strpos($data, 'admin_') === 0) {
        // کال‌بک‌های پنل مدیریت
        if (strpos($data, 'admin_create_exam') === 0 || 
            strpos($data, 'admin_cancel_exam_creation') === 0 || 
            strpos($data, 'admin_list_exams') === 0 || 
            strpos($data, 'admin_view_exam_') === 0 || 
            strpos($data, 'admin_get_exam_link_') === 0 || 
            strpos($data, 'admin_exam_panel') === 0) {
            
            // پردازش کال‌بک‌های مربوط به مدیریت آزمون‌ها
            handleExamAdminCallback($callback);
        } else {
            // سایر کال‌بک‌های مدیریتی
            handleAdminCallback($callback);
        }
    } elseif (strpos($data, 'exam_') === 0) {
        // کال‌بک‌های آزمون
        handleExamCallback($callback);
    } elseif (strpos($data, 'back_to_main') === 0) {
        // کال‌بک بازگشت به منوی اصلی
        showMainMenu($chat_id);
    } elseif (strpos($data, 'inline_') === 0) {
        // کال‌بک انتخاب دکمه اینلاین
        $inlineButtonId = str_replace('inline_', '', $data);
        
        // دریافت اطلاعات دکمه اینلاین
        $stmt = $pdo->prepare("SELECT ib.*, mb.title as main_title 
                              FROM inline_buttons ib 
                              JOIN main_buttons mb ON ib.main_button_id = mb.id 
                              WHERE ib.id = :id");
        $stmt->execute([':id' => $inlineButtonId]);
        $inlineButton = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($inlineButton) {
            // نمایش توضیحات دکمه اینلاین
            $message = "*{$inlineButton['main_title']} > {$inlineButton['title']}*\n\n";
            if (!empty($inlineButton['description'])) {
                $message .= "{$inlineButton['description']}\n\n";
            }
            
            // دریافت فایل‌های مرتبط با این دکمه اینلاین
            $stmt = $pdo->prepare("SELECT * FROM files WHERE inline_button_id = :id");
            $stmt->execute([':id' => $inlineButtonId]);
            $files = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            if (!empty($files)) {
                foreach ($files as $file) {
                    // ارسال فایل‌ها
                    switch ($file['file_type']) {
                        case 'document':
                            sendDocument($chat_id, $file['file_path'], $file['caption']);
                            break;
                        case 'photo':
                            sendPhoto($chat_id, $file['file_path'], $file['caption']);
                            break;
                        case 'video':
                            sendVideo($chat_id, $file['file_path'], $file['caption']);
                            break;
                        case 'audio':
                            sendAudio($chat_id, $file['file_path'], $file['caption']);
                            break;
                    }
                }
            } else {
                sendMessage($chat_id, "هیچ فایلی برای این بخش یافت نشد.");
            }
            
            // دکمه بازگشت به منوی اصلی دکمه‌های اصلی
            $back_button = [
                [['text' => '🔙 بازگشت', 'callback_data' => 'back_to_main']]
            ];
            
            $reply_markup = ['inline_keyboard' => $back_button];
            sendMessage($chat_id, $message, $reply_markup, "Markdown");
        } else {
            sendMessage($chat_id, "دکمه مورد نظر یافت نشد.");
        }
    }
}

// ثبت لاگ پایان اجرای اسکریپت
logger()->info("پایان اجرای ربات تلگرام", [
    'time' => date('Y-m-d H:i:s'),
    'script' => 'bot.php',
    'memory_usage' => memory_get_usage(true) / 1024 / 1024 . ' MB',
    'execution_time' => microtime(true) - $_SERVER['REQUEST_TIME_FLOAT'] . ' seconds'
]);

?>
