<?php
// admin_functions.php
require_once 'config.php';
require_once 'telegram_api.php';
require_once 'logger.php';

/**
 * نمایش پنل مدیریت برای ادمین
 */
function showAdminPanel($chat_id) {
    global $pdo;
    
    logger()->info("نمایش پنل مدیریت برای ادمین", [
        'chat_id' => $chat_id,
        'function' => 'showAdminPanel'
    ]);
    
    // ایجاد دکمه‌های اینلاین برای پنل مدیریت
    $inline_keyboard = [
        [
            ['text' => '📊 داشبورد و آمار', 'callback_data' => 'admin_dashboard']
        ],
        [
            ['text' => '➕ افزودن دکمه اصلی', 'callback_data' => 'admin_add_main_button']
        ],
        [
            ['text' => '➕ افزودن دکمه اینلاین', 'callback_data' => 'admin_add_inline_button']
        ],
        [
            ['text' => '📂 آپلود فایل', 'callback_data' => 'admin_upload_file']
        ],
        [
            ['text' => '📝 مدیریت دکمه‌های اصلی', 'callback_data' => 'admin_manage_main_buttons']
        ],
        [
            ['text' => '📝 مدیریت دکمه‌های اینلاین', 'callback_data' => 'admin_manage_inline_buttons']
        ],
        [
            ['text' => '📝 مدیریت فایل‌ها', 'callback_data' => 'admin_manage_files']
        ],
        [
            ['text' => '👥 مدیریت کاربران', 'callback_data' => 'admin_manage_users']
        ],
        [
            ['text' => '📨 فایل‌های پیشنهادی', 'callback_data' => 'admin_suggested_files']
        ],
        [
            ['text' => '📋 مدیریت آزمون‌ها', 'callback_data' => 'admin_manage_exams']
        ]
    ];
    
    $reply_markup = ['inline_keyboard' => $inline_keyboard];
    sendMessage($chat_id, "✅ به پنل مدیریت ربات خوش آمدید.\n\nلطفاً یکی از گزینه‌های زیر را انتخاب کنید:", $reply_markup);
}

/**
 * مدیریت کال‌بک‌های مربوط به پنل مدیریت
 */
function handleAdminCallback($callback) {
    global $pdo;
    
    $chat_id = $callback['message']['chat']['id'];
    $message_id = $callback['message']['message_id'];
    $data = $callback['data'];
    
    logger()->info("مدیریت کال‌بک ادمین", [
        'chat_id' => $chat_id,
        'callback_data' => $data,
        'function' => 'handleAdminCallback'
    ]);
    
    // پاسخ اولیه به کال‌بک‌کوئری
    apiRequest('answerCallbackQuery', ['callback_query_id' => $callback['id']]);
    
    switch ($data) {
        case 'admin_dashboard':
            logger()->debug("نمایش داشبورد آمار", ['chat_id' => $chat_id]);
            
            // نمایش داشبورد آماری
            $totalUsers = $pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();
            $activeUsers = $pdo->query("SELECT COUNT(*) FROM users WHERE status = 'active'")->fetchColumn();
            $blockedUsers = $pdo->query("SELECT COUNT(*) FROM users WHERE status = 'blocked'")->fetchColumn();
            $totalMainButtons = $pdo->query("SELECT COUNT(*) FROM main_buttons")->fetchColumn();
            $totalInlineButtons = $pdo->query("SELECT COUNT(*) FROM inline_buttons")->fetchColumn();
            $totalFiles = $pdo->query("SELECT COUNT(*) FROM files")->fetchColumn();
            $pendingSuggestions = $pdo->query("SELECT COUNT(*) FROM suggested_files WHERE status = 'pending'")->fetchColumn();
            $totalExams = $pdo->query("SELECT COUNT(*) FROM exams")->fetchColumn();
            $activeExams = $pdo->query("SELECT COUNT(*) FROM exams WHERE start_time <= NOW() AND end_time >= NOW()")->fetchColumn();
            
            logger()->debug("آمار داشبورد", [
                'total_users' => $totalUsers,
                'active_users' => $activeUsers,
                'blocked_users' => $blockedUsers,
                'total_main_buttons' => $totalMainButtons,
                'total_inline_buttons' => $totalInlineButtons,
                'total_files' => $totalFiles,
                'pending_suggestions' => $pendingSuggestions,
                'total_exams' => $totalExams,
                'active_exams' => $activeExams
            ]);
            
            $stats = "📊 *داشبورد مدیریت*\n\n".
                     "👥 *کاربران:*\n".
                     "- کل کاربران: $totalUsers\n".
                     "- کاربران فعال: $activeUsers\n".
                     "- کاربران مسدود: $blockedUsers\n\n".
                     "🔘 *دکمه‌ها:*\n".
                     "- دکمه‌های اصلی: $totalMainButtons\n".
                     "- دکمه‌های اینلاین: $totalInlineButtons\n\n".
                     "📂 *فایل‌ها:*\n".
                     "- کل فایل‌ها: $totalFiles\n".
                     "- فایل‌های پیشنهادی در انتظار: $pendingSuggestions\n\n".
                     "📝 *آزمون‌ها:*\n".
                     "- کل آزمون‌ها: $totalExams\n".
                     "- آزمون‌های فعال: $activeExams\n";
                     
            $backButton = [
                [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']]
            ];
            
            $reply_markup = ['inline_keyboard' => $backButton];
            sendMessage($chat_id, $stats, $reply_markup, "Markdown");
            break;
            
        case 'admin_add_main_button':
            // فرم افزودن دکمه اصلی
            $formText = "برای افزودن دکمه اصلی، متن دکمه را با فرمت زیر ارسال کنید:\n\n".
                         "/add_main_button عنوان دکمه";
            
            $backButton = [
                [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']]
            ];
            
            $reply_markup = ['inline_keyboard' => $backButton];
            sendMessage($chat_id, $formText, $reply_markup);
            break;

        case 'admin_add_inline_button':
            // نمایش لیست دکمه‌های اصلی برای انتخاب
            $stmt = $pdo->query("SELECT id, title FROM main_buttons ORDER BY title");
            $mainButtons = [];
            
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $mainButtons[] = [
                    ['text' => $row['title'], 'callback_data' => 'admin_select_main_' . $row['id']]
                ];
            }
            
            $mainButtons[] = [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']];
            
            $reply_markup = ['inline_keyboard' => $mainButtons];
            sendMessage($chat_id, "لطفاً دکمه اصلی مورد نظر برای افزودن دکمه اینلاین را انتخاب کنید:", $reply_markup);
            break;
        
        case 'admin_upload_file':
            // نمایش لیست دکمه‌های اینلاین برای انتخاب
            $stmt = $pdo->query("SELECT ib.id, ib.title, mb.title as main_title 
                                FROM inline_buttons ib 
                                JOIN main_buttons mb ON ib.main_button_id = mb.id 
                                ORDER BY mb.title, ib.title");
            $inlineButtons = [];
            $currentMainTitle = '';
            
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                if ($currentMainTitle != $row['main_title']) {
                    $currentMainTitle = $row['main_title'];
                    $inlineButtons[] = [['text' => "📌 {$row['main_title']}", 'callback_data' => 'admin_no_action']];
                }
                
                $inlineButtons[] = [['text' => $row['title'], 'callback_data' => 'admin_select_inline_' . $row['id']]];
            }
            
            $inlineButtons[] = [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']];
            
            $reply_markup = ['inline_keyboard' => $inlineButtons];
            sendMessage($chat_id, "لطفاً دکمه اینلاین مورد نظر برای آپلود فایل را انتخاب کنید:", $reply_markup);
            break;
            
        case 'admin_manage_main_buttons':
            // نمایش لیست دکمه‌های اصلی برای مدیریت
            $stmt = $pdo->query("SELECT id, title FROM main_buttons ORDER BY title");
            $mainButtons = [];
            
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $mainButtons[] = [
                    ['text' => $row['title'], 'callback_data' => 'admin_edit_main_' . $row['id']],
                    ['text' => '❌', 'callback_data' => 'admin_delete_main_' . $row['id']]
                ];
            }
            
            $mainButtons[] = [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']];
            
            $reply_markup = ['inline_keyboard' => $mainButtons];
            sendMessage($chat_id, "مدیریت دکمه‌های اصلی:\n\nبرای ویرایش روی عنوان دکمه، و برای حذف روی ❌ کلیک کنید.", $reply_markup);
            break;
            
        case 'admin_manage_inline_buttons':
            // نمایش لیست دکمه‌های اینلاین برای مدیریت
            $stmt = $pdo->query("SELECT ib.id, ib.title, mb.title as main_title 
                                FROM inline_buttons ib 
                                JOIN main_buttons mb ON ib.main_button_id = mb.id 
                                ORDER BY mb.title, ib.title");
            $inlineButtons = [];
            $currentMainTitle = '';
            
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                if ($currentMainTitle != $row['main_title']) {
                    $currentMainTitle = $row['main_title'];
                    $inlineButtons[] = [['text' => "📌 {$row['main_title']}", 'callback_data' => 'admin_no_action']];
                }
                
                $inlineButtons[] = [
                    ['text' => $row['title'], 'callback_data' => 'admin_edit_inline_' . $row['id']],
                    ['text' => '❌', 'callback_data' => 'admin_delete_inline_' . $row['id']]
                ];
            }
            
            $inlineButtons[] = [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']];
            
            $reply_markup = ['inline_keyboard' => $inlineButtons];
            sendMessage($chat_id, "مدیریت دکمه‌های اینلاین:\n\nبرای ویرایش روی عنوان دکمه، و برای حذف روی ❌ کلیک کنید.", $reply_markup);
            break;
            
        case 'admin_manage_files':
            // نمایش لیست فایل‌ها برای مدیریت
            $stmt = $pdo->query("SELECT f.id, f.file_type, f.caption, ib.title as inline_title, mb.title as main_title 
                                FROM files f 
                                JOIN inline_buttons ib ON f.inline_button_id = ib.id 
                                JOIN main_buttons mb ON ib.main_button_id = mb.id 
                                ORDER BY mb.title, ib.title, f.id");
            $files = [];
            $currentTitle = '';
            
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $title = "{$row['main_title']} > {$row['inline_title']}";
                
                if ($currentTitle != $title) {
                    $currentTitle = $title;
                    $files[] = [['text' => "📌 $title", 'callback_data' => 'admin_no_action']];
                }
                
                $caption = $row['caption'] ? (mb_strlen($row['caption']) > 20 ? mb_substr($row['caption'], 0, 17) . '...' : $row['caption']) : 'بدون عنوان';
                
                $files[] = [
                    ['text' => "[{$row['file_type']}] $caption", 'callback_data' => 'admin_edit_file_' . $row['id']],
                    ['text' => '❌', 'callback_data' => 'admin_delete_file_' . $row['id']]
                ];
            }
            
            $files[] = [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']];
            
            $reply_markup = ['inline_keyboard' => $files];
            sendMessage($chat_id, "مدیریت فایل‌ها:\n\nبرای ویرایش روی عنوان فایل، و برای حذف روی ❌ کلیک کنید.", $reply_markup);
            break;
            
        case 'admin_manage_users':
            // نمایش لیست کاربران برای مدیریت
            $page = 0;
            $limit = 10;
            $offset = $page * $limit;
            
            $stmt = $pdo->query("SELECT id, telegram_id, username, full_name, status 
                                FROM users 
                                ORDER BY last_usage DESC 
                                LIMIT $limit OFFSET $offset");
            $users = [];
            
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $name = $row['full_name'] ? $row['full_name'] : ($row['username'] ? $row['username'] : $row['telegram_id']);
                $status = $row['status'] == 'active' ? '✅' : '❌';
                
                $users[] = [
                    ['text' => "$name [$status]", 'callback_data' => 'admin_user_info_' . $row['id']],
                    ['text' => $row['status'] == 'active' ? '🚫 مسدود' : '✅ فعال', 'callback_data' => 'admin_toggle_user_' . $row['id']]
                ];
            }
            
            // دکمه‌های ناوبری
            $navButtons = [];
            
            $totalUsers = $pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();
            $totalPages = ceil($totalUsers / $limit);
            
            if ($page > 0) {
                $navButtons[] = ['text' => '◀️ قبلی', 'callback_data' => 'admin_users_page_' . ($page - 1)];
            }
            
            if ($page < $totalPages - 1) {
                $navButtons[] = ['text' => 'بعدی ▶️', 'callback_data' => 'admin_users_page_' . ($page + 1)];
            }
            
            if (!empty($navButtons)) {
                $users[] = $navButtons;
            }
            
            $users[] = [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']];
            
            $reply_markup = ['inline_keyboard' => $users];
            sendMessage($chat_id, "مدیریت کاربران:\n\nصفحه " . ($page + 1) . " از $totalPages", $reply_markup);
            break;
            
        case 'admin_suggested_files':
            // نمایش لیست فایل‌های پیشنهادی
            $stmt = $pdo->query("SELECT sf.id, sf.file_url, sf.status, sf.submission_time, 
                                        u.telegram_id, u.username, u.full_name 
                                FROM suggested_files sf 
                                JOIN users u ON sf.user_id = u.id 
                                WHERE sf.status = 'pending' 
                                ORDER BY sf.submission_time DESC");
            $suggestedFiles = [];
            
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $user = $row['full_name'] ? $row['full_name'] : ($row['username'] ? $row['username'] : $row['telegram_id']);
                $time = date('Y-m-d H:i', strtotime($row['submission_time']));
                
                $suggestedFiles[] = [
                    ['text' => "👤 $user | 🕒 $time", 'callback_data' => 'admin_view_suggested_' . $row['id']]
                ];
                
                $suggestedFiles[] = [
                    ['text' => '✅ تایید', 'callback_data' => 'admin_approve_suggested_' . $row['id']],
                    ['text' => '❌ رد', 'callback_data' => 'admin_reject_suggested_' . $row['id']]
                ];
            }
            
            if (empty($suggestedFiles)) {
                $suggestedFiles[] = [['text' => 'هیچ فایل پیشنهادی در انتظار تایید وجود ندارد.', 'callback_data' => 'admin_no_action']];
            }
            
            $suggestedFiles[] = [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']];
            
            $reply_markup = ['inline_keyboard' => $suggestedFiles];
            sendMessage($chat_id, "فایل‌های پیشنهادی در انتظار تایید:", $reply_markup);
            break;
            
        case 'admin_manage_exams':
            // نمایش پنل مدیریت آزمون‌ها برای ادمین
            showAdminExamPanel($chat_id);
            break;
            
        case 'admin_back_to_panel':
            // بازگشت به پنل اصلی
            showAdminPanel($chat_id);
            break;
            
        case 'admin_no_action':
            // هیچ عملیاتی انجام نشود
            break;
            
        default:
            // بررسی الگوهای پیچیده‌تر
            if (strpos($data, 'admin_select_main_') === 0) {
                $mainButtonId = str_replace('admin_select_main_', '', $data);
                $stmt = $pdo->prepare("SELECT title FROM main_buttons WHERE id = :id");
                $stmt->execute([':id' => $mainButtonId]);
                $mainButtonTitle = $stmt->fetchColumn();
                
                $formText = "برای افزودن دکمه اینلاین به دکمه اصلی \"$mainButtonTitle\"، متن زیر را با فرمت مشخص شده ارسال کنید:\n\n".
                             "/add_inline_button $mainButtonId عنوان دکمه اینلاین | توضیحات (اختیاری)";
                
                $backButton = [
                    [['text' => '🔙 بازگشت به انتخاب دکمه اصلی', 'callback_data' => 'admin_add_inline_button']],
                    [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']]
                ];
                
                $reply_markup = ['inline_keyboard' => $backButton];
                sendMessage($chat_id, $formText, $reply_markup);
            }
            elseif (strpos($data, 'admin_select_inline_') === 0) {
                $inlineButtonId = str_replace('admin_select_inline_', '', $data);
                $stmt = $pdo->prepare("
                    SELECT ib.title as inline_title, mb.title as main_title 
                    FROM inline_buttons ib 
                    JOIN main_buttons mb ON ib.main_button_id = mb.id 
                    WHERE ib.id = :id
                ");
                $stmt->execute([':id' => $inlineButtonId]);
                $button = $stmt->fetch(PDO::FETCH_ASSOC);
                
                $formText = "برای آپلود فایل به دکمه اینلاین \"" . $button['inline_title'] . "\" (در " . $button['main_title'] . ")، لطفاً نوع فایل را انتخاب کنید:";
                
                $fileTypes = [
                    [
                        ['text' => '📄 سند (Document)', 'callback_data' => 'admin_upload_document_' . $inlineButtonId],
                        ['text' => '🎬 ویدیو (Video)', 'callback_data' => 'admin_upload_video_' . $inlineButtonId]
                    ],
                    [
                        ['text' => '🎵 صوت (Audio)', 'callback_data' => 'admin_upload_audio_' . $inlineButtonId],
                        ['text' => '📝 متن (Text)', 'callback_data' => 'admin_upload_text_' . $inlineButtonId]
                    ],
                    [
                        ['text' => '🔙 بازگشت به انتخاب دکمه اینلاین', 'callback_data' => 'admin_upload_file']
                    ],
                    [
                        ['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']
                    ]
                ];
                
                $reply_markup = ['inline_keyboard' => $fileTypes];
                sendMessage($chat_id, $formText, $reply_markup);
            }
            elseif (strpos($data, 'admin_upload_') === 0 && strpos($data, '_', 13) !== false) {
                // دستور آپلود فایل: admin_upload_{file_type}_{inline_button_id}
                $parts = explode('_', $data);
                if (count($parts) >= 4) {
                    $fileType = $parts[2];
                    $inlineButtonId = $parts[3];
                    
                    $stmt = $pdo->prepare("
                        SELECT ib.title as inline_title, mb.title as main_title 
                        FROM inline_buttons ib 
                        JOIN main_buttons mb ON ib.main_button_id = mb.id 
                        WHERE ib.id = :id
                    ");
                    $stmt->execute([':id' => $inlineButtonId]);
                    $button = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    $fileTypeText = '';
                    switch ($fileType) {
                        case 'document':
                            $fileTypeText = 'سند (Document)';
                            break;
                        case 'video':
                            $fileTypeText = 'ویدیو (Video)';
                            break;
                        case 'audio':
                            $fileTypeText = 'صوت (Audio)';
                            break;
                        case 'text':
                            $fileTypeText = 'متن (Text)';
                            break;
                    }
                    
                    $formText = "برای آپلود فایل $fileTypeText به دکمه اینلاین \"" . $button['inline_title'] . "\" (در " . $button['main_title'] . ")، دستور زیر را ارسال کنید:\n\n".
                                 "/upload_file $inlineButtonId $fileType | عنوان فایل (اختیاری)\n\n".
                                 "سپس فایل مورد نظر را ارسال کنید.";
                    
                    $backButton = [
                        [['text' => '🔙 بازگشت به انتخاب نوع فایل', 'callback_data' => 'admin_select_inline_' . $inlineButtonId]],
                        [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']]
                    ];
                    
                    $reply_markup = ['inline_keyboard' => $backButton];
                    sendMessage($chat_id, $formText, $reply_markup);
                }
            }
            elseif (strpos($data, 'admin_edit_main_') === 0) {
                $mainButtonId = str_replace('admin_edit_main_', '', $data);
                $stmt = $pdo->prepare("SELECT title FROM main_buttons WHERE id = :id");
                $stmt->execute([':id' => $mainButtonId]);
                $mainButtonTitle = $stmt->fetchColumn();
                
                $formText = "برای ویرایش دکمه اصلی \"$mainButtonTitle\"، متن زیر را با فرمت مشخص شده ارسال کنید:\n\n".
                             "/edit_main_button $mainButtonId عنوان جدید";
                
                $backButton = [
                    [['text' => '🔙 بازگشت به مدیریت دکمه‌های اصلی', 'callback_data' => 'admin_manage_main_buttons']],
                    [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']]
                ];
                
                $reply_markup = ['inline_keyboard' => $backButton];
                sendMessage($chat_id, $formText, $reply_markup);
            }
            elseif (strpos($data, 'admin_delete_main_') === 0) {
                $mainButtonId = str_replace('admin_delete_main_', '', $data);
                $stmt = $pdo->prepare("SELECT title FROM main_buttons WHERE id = :id");
                $stmt->execute([':id' => $mainButtonId]);
                $mainButtonTitle = $stmt->fetchColumn();
                
                $confirmButtons = [
                    [
                        ['text' => '✅ بله، حذف شود', 'callback_data' => 'admin_confirm_delete_main_' . $mainButtonId],
                        ['text' => '❌ خیر', 'callback_data' => 'admin_manage_main_buttons']
                    ],
                    [
                        ['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']
                    ]
                ];
                
                $reply_markup = ['inline_keyboard' => $confirmButtons];
                sendMessage($chat_id, "آیا از حذف دکمه اصلی \"$mainButtonTitle\" اطمینان دارید؟\n\n⚠️ توجه: با حذف این دکمه، تمام دکمه‌های اینلاین و فایل‌های مرتبط نیز حذف خواهند شد.", $reply_markup);
            }
            elseif (strpos($data, 'admin_confirm_delete_main_') === 0) {
                $mainButtonId = str_replace('admin_confirm_delete_main_', '', $data);
                
                // حذف دکمه اصلی و تمام موارد وابسته به آن
                $stmt = $pdo->prepare("DELETE FROM main_buttons WHERE id = :id");
                if ($stmt->execute([':id' => $mainButtonId])) {
                    sendMessage($chat_id, "✅ دکمه اصلی و تمام موارد وابسته به آن با موفقیت حذف شد.");
                } else {
                    sendMessage($chat_id, "❌ خطا در حذف دکمه اصلی.");
                }
                
                // نمایش مجدد لیست دکمه‌های اصلی
                $data = 'admin_manage_main_buttons';
                handleAdminCallback($callback);
            }
            // پردازش کال‌بک‌های مدیریت آزمون
            elseif ($data === 'admin_add_exam') {
                $formText = "برای افزودن آزمون جدید، متن زیر را با فرمت مشخص شده ارسال کنید:\n\n".
                             "/add_exam عنوان آزمون | توضیحات آزمون | زمان پایان (YYYY-MM-DD HH:MM:SS)";
                
                $backButton = [
                    [['text' => '🔙 بازگشت به مدیریت آزمون‌ها', 'callback_data' => 'admin_manage_exams']],
                    [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']]
                ];
                
                $reply_markup = ['inline_keyboard' => $backButton];
                sendMessage($chat_id, $formText, $reply_markup);
            }
            elseif ($data === 'admin_manage_exams') {
                // نمایش لیست آزمون‌ها برای مدیریت
                $stmt = $pdo->query("SELECT id, title, start_time, end_time FROM exams ORDER BY start_time DESC");
                $exams = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if (empty($exams)) {
                    $examButtons = [
                        [['text' => '➕ افزودن آزمون جدید', 'callback_data' => 'admin_add_exam']],
                        [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']]
                    ];
                    
                    $reply_markup = ['inline_keyboard' => $examButtons];
                    sendMessage($chat_id, "هیچ آزمونی یافت نشد. برای افزودن آزمون جدید کلیک کنید.", $reply_markup);
                } else {
                    $examButtons = [];
                    foreach ($exams as $exam) {
                        $status = (strtotime($exam['end_time']) > time()) ? '🟢' : '🔴';
                        $examButtons[] = [
                            ['text' => "$status {$exam['title']}", 'callback_data' => 'admin_view_exam_' . $exam['id']]
                        ];
                        $examButtons[] = [
                            ['text' => '➕ افزودن سوال', 'callback_data' => 'admin_add_question_' . $exam['id']],
                            ['text' => '❌ حذف', 'callback_data' => 'admin_delete_exam_' . $exam['id']]
                        ];
                    }
                    
                    $examButtons[] = [['text' => '➕ افزودن آزمون جدید', 'callback_data' => 'admin_add_exam']];
                    $examButtons[] = [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']];
                    
                    $reply_markup = ['inline_keyboard' => $examButtons];
                    sendMessage($chat_id, "مدیریت آزمون‌ها:\n\nبرای مشاهده جزئیات روی عنوان آزمون، برای افزودن سوال روی '➕ افزودن سوال'، و برای حذف روی '❌ حذف' کلیک کنید.", $reply_markup);
                }
            }
            elseif ($data === 'admin_exam_results') {
                // نمایش گزارش نتایج آزمون‌ها
                $stmt = $pdo->query("SELECT id, title FROM exams ORDER BY start_time DESC");
                $exams = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if (empty($exams)) {
                    $backButton = [
                        [['text' => '🔙 بازگشت به مدیریت آزمون‌ها', 'callback_data' => 'admin_manage_exams']],
                        [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']]
                    ];
                    
                    $reply_markup = ['inline_keyboard' => $backButton];
                    sendMessage($chat_id, "هیچ آزمونی یافت نشد.", $reply_markup);
                } else {
                    $examButtons = [];
                    foreach ($exams as $exam) {
                        $examButtons[] = [
                            ['text' => $exam['title'], 'callback_data' => 'admin_exam_result_' . $exam['id']]
                        ];
                    }
                    
                    $examButtons[] = [['text' => '🔙 بازگشت به مدیریت آزمون‌ها', 'callback_data' => 'admin_manage_exams']];
                    $examButtons[] = [['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']];
                    
                    $reply_markup = ['inline_keyboard' => $examButtons];
                    sendMessage($chat_id, "لطفاً آزمون مورد نظر برای مشاهده نتایج را انتخاب کنید:", $reply_markup);
                }
            }
            // سایر موارد پیچیده را اینجا اضافه کنید
    }
}

/**
 * نمایش پنل مدیریت آزمون‌ها برای ادمین
 */
function showAdminExamPanel($chat_id) {
    global $pdo;
    
    logger()->info("نمایش پنل مدیریت آزمون‌ها برای ادمین", [
        'chat_id' => $chat_id,
        'function' => 'showAdminExamPanel'
    ]);
    
    // ایجاد دکمه‌های اینلاین برای پنل مدیریت آزمون
    $inline_keyboard = [
        [
            ['text' => '➕ ایجاد آزمون جدید', 'callback_data' => 'admin_create_exam']
        ],
        [
            ['text' => '📋 لیست آزمون‌های فعال', 'callback_data' => 'admin_list_exams_active']
        ],
        [
            ['text' => '📋 لیست همه آزمون‌ها', 'callback_data' => 'admin_list_exams_all']
        ],
        [
            ['text' => '🔙 بازگشت به پنل مدیریت', 'callback_data' => 'admin_back_to_panel']
        ]
    ];
    
    $reply_markup = ['inline_keyboard' => $inline_keyboard];
    sendMessage($chat_id, "✅ *مدیریت آزمون‌ها*\n\nلطفاً یکی از گزینه‌های زیر را انتخاب کنید:", $reply_markup, "Markdown");
}

/**
 * مدیریت کال‌بک‌های مربوط به مدیریت آزمون‌ها
 */
function handleExamAdminCallback($callback) {
    global $pdo;
    
    $chat_id = $callback['message']['chat']['id'];
    $message_id = $callback['message']['message_id'];
    $data = $callback['data'];
    
    logger()->info("مدیریت کال‌بک مدیریت آزمون", [
        'chat_id' => $chat_id,
        'callback_data' => $data,
        'function' => 'handleExamAdminCallback'
    ]);
    
    // پاسخ اولیه به کال‌بک‌کوئری
    apiRequest('answerCallbackQuery', ['callback_query_id' => $callback['id']]);
    
    if ($data === 'admin_create_exam') {
        // شروع فرآیند ایجاد آزمون جدید
        $message = "*ایجاد آزمون جدید*\n\n";
        $message .= "برای ایجاد آزمون جدید، اطلاعات زیر را وارد کنید:\n\n";
        $message .= "1️⃣ عنوان آزمون\n";
        $message .= "2️⃣ توضیحات آزمون (اختیاری)\n";
        $message .= "3️⃣ تاریخ و زمان شروع (مثال: 1400/01/01 14:30)\n";
        $message .= "4️⃣ تاریخ و زمان پایان (مثال: 1400/01/01 16:30)\n\n";
        $message .= "لطفاً عنوان آزمون را وارد کنید:";
        
        $_SESSION['exam_creation'][$chat_id] = [
            'step' => 1,
            'title' => '',
            'description' => '',
            'start_time' => '',
            'end_time' => ''
        ];
        
        $cancel_button = [
            [['text' => '🔙 انصراف و بازگشت', 'callback_data' => 'admin_cancel_exam_creation']]
        ];
        
        $reply_markup = ['inline_keyboard' => $cancel_button];
        sendMessage($chat_id, $message, $reply_markup, "Markdown");
    }
    elseif ($data === 'admin_cancel_exam_creation') {
        // انصراف از ایجاد آزمون
        if (isset($_SESSION['exam_creation'][$chat_id])) {
            unset($_SESSION['exam_creation'][$chat_id]);
        }
        
        sendMessage($chat_id, "❌ فرآیند ایجاد آزمون لغو شد.");
        showAdminExamPanel($chat_id);
    }
    elseif ($data === 'admin_list_exams_active' || $data === 'admin_list_exams_all') {
        // نمایش لیست آزمون‌ها
        $active_only = ($data === 'admin_list_exams_active');
        
        if ($active_only) {
            $stmt = $pdo->prepare("
                SELECT * FROM exams 
                WHERE start_time <= NOW() AND end_time >= NOW()
                ORDER BY start_time
            ");
        } else {
            $stmt = $pdo->prepare("SELECT * FROM exams ORDER BY start_time DESC");
        }
        
        $stmt->execute();
        $exams = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (empty($exams)) {
            $message = $active_only ? 
                "❌ در حال حاضر هیچ آزمون فعالی وجود ندارد." : 
                "❌ هیچ آزمونی یافت نشد.";
                
            $back_button = [
                [['text' => '🔙 بازگشت', 'callback_data' => 'admin_exam_panel']]
            ];
            
            $reply_markup = ['inline_keyboard' => $back_button];
            sendMessage($chat_id, $message, $reply_markup);
            return;
        }
        
        $message = $active_only ? 
            "*📋 لیست آزمون‌های فعال:*\n\n" : 
            "*📋 لیست تمام آزمون‌ها:*\n\n";
        
        $exam_buttons = [];
        foreach ($exams as $index => $exam) {
            $exam_status = '';
            $now = new \DateTime();
            $start = new \DateTime($exam['start_time']);
            $end = new \DateTime($exam['end_time']);
            
            if ($now < $start) {
                $exam_status = '⏳ منتظر شروع';
            } elseif ($now > $end) {
                $exam_status = '✅ پایان یافته';
            } else {
                $exam_status = '🔵 در حال اجرا';
            }
            
            $message .= ($index + 1) . ". *{$exam['title']}*\n";
            $message .= "   🕒 شروع: " . date('Y/m/d H:i', strtotime($exam['start_time'])) . "\n";
            $message .= "   🕒 پایان: " . date('Y/m/d H:i', strtotime($exam['end_time'])) . "\n";
            $message .= "   📊 وضعیت: $exam_status\n\n";
            
            $exam_buttons[] = [
                ['text' => ($index + 1) . ". " . $exam['title'], 'callback_data' => 'admin_view_exam_' . $exam['id']]
            ];
        }
        
        $exam_buttons[] = [['text' => '🔙 بازگشت', 'callback_data' => 'admin_exam_panel']];
        
        $reply_markup = ['inline_keyboard' => $exam_buttons];
        sendMessage($chat_id, $message, $reply_markup, "Markdown");
    }
    elseif (strpos($data, 'admin_view_exam_') === 0) {
        // نمایش جزئیات آزمون و گزینه‌های مدیریت آن
        $exam_id = str_replace('admin_view_exam_', '', $data);
        
        $stmt = $pdo->prepare("SELECT * FROM exams WHERE id = :id");
        $stmt->execute([':id' => $exam_id]);
        $exam = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$exam) {
            sendMessage($chat_id, "❌ آزمون مورد نظر یافت نشد.");
            return;
        }
        
        // دریافت تعداد سوالات
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM questions WHERE exam_id = :exam_id");
        $stmt->execute([':exam_id' => $exam_id]);
        $questions_count = $stmt->fetchColumn();
        
        // دریافت تعداد شرکت‌کنندگان
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM results WHERE exam_id = :exam_id");
        $stmt->execute([':exam_id' => $exam_id]);
        $participants_count = $stmt->fetchColumn();
        
        $exam_status = '';
        $now = new \DateTime();
        $start = new \DateTime($exam['start_time']);
        $end = new \DateTime($exam['end_time']);
        
        if ($now < $start) {
            $exam_status = '⏳ منتظر شروع';
        } elseif ($now > $end) {
            $exam_status = '✅ پایان یافته';
        } else {
            $exam_status = '🔵 در حال اجرا';
        }
        
        $message = "*🔍 جزئیات آزمون:*\n\n";
        $message .= "📝 *عنوان:* {$exam['title']}\n";
        
        if (!empty($exam['description'])) {
            $message .= "📋 *توضیحات:* {$exam['description']}\n";
        }
        
        $message .= "🕒 *شروع:* " . date('Y/m/d H:i', strtotime($exam['start_time'])) . "\n";
        $message .= "🕒 *پایان:* " . date('Y/m/d H:i', strtotime($exam['end_time'])) . "\n";
        $message .= "📊 *وضعیت:* $exam_status\n";
        $message .= "❓ *تعداد سوالات:* $questions_count\n";
        $message .= "👥 *تعداد شرکت‌کنندگان:* $participants_count\n";
        
        // ایجاد دکمه‌های مدیریت آزمون
        $manage_buttons = [
            [
                ['text' => '➕ افزودن سوال', 'callback_data' => 'admin_add_question_' . $exam_id],
                ['text' => '📋 مشاهده سوالات', 'callback_data' => 'admin_view_questions_' . $exam_id]
            ],
            [
                ['text' => '✏️ ویرایش آزمون', 'callback_data' => 'admin_edit_exam_' . $exam_id],
                ['text' => '❌ حذف آزمون', 'callback_data' => 'admin_confirm_delete_exam_' . $exam_id]
            ],
            [
                ['text' => '🔗 لینک آزمون', 'callback_data' => 'admin_get_exam_link_' . $exam_id]
            ],
            [
                ['text' => '🔙 بازگشت', 'callback_data' => 'admin_list_exams_all']
            ]
        ];
        
        $reply_markup = ['inline_keyboard' => $manage_buttons];
        sendMessage($chat_id, $message, $reply_markup, "Markdown");
    }
    elseif (strpos($data, 'admin_get_exam_link_') === 0) {
        // نمایش لینک آزمون
        $exam_id = str_replace('admin_get_exam_link_', '', $data);
        
        $stmt = $pdo->prepare("SELECT title FROM exams WHERE id = :id");
        $stmt->execute([':id' => $exam_id]);
        $exam_title = $stmt->fetchColumn();
        
        if (!$exam_title) {
            sendMessage($chat_id, "❌ آزمون مورد نظر یافت نشد.");
            return;
        }
        
        // ساخت لینک آزمون برای دمو
        $base_url = "https://" . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']);
        if (substr($base_url, -1) !== '/') {
            $base_url .= '/';
        }
        
        $exam_link = $base_url . "exam.php?exam_id={$exam_id}&demo=1";
        
        $message = "*🔗 لینک آزمون:* {$exam_title}\n\n";
        $message .= $exam_link . "\n\n";
        $message .= "این لینک برای دسترسی مستقیم به آزمون است. کاربران نیاز به ورود به سیستم ندارند.\n";
        $message .= "می‌توانید این لینک را برای کاربران ارسال کنید تا مستقیماً وارد آزمون شوند.";
        
        $back_button = [
            [['text' => '🔙 بازگشت', 'callback_data' => 'admin_view_exam_' . $exam_id]]
        ];
        
        $reply_markup = ['inline_keyboard' => $back_button];
        sendMessage($chat_id, $message, $reply_markup, "Markdown");
    }
    elseif ($data === 'admin_exam_panel') {
        // بازگشت به پنل مدیریت آزمون‌ها
        showAdminExamPanel($chat_id);
    }
    elseif ($data === 'admin_back_to_panel') {
        // بازگشت به پنل مدیریت اصلی
        showAdminPanel($chat_id);
    }
}

/**
 * پردازش آپلود فایل
 */
function processFileUpload($chat_id, $inline_button_id, $file_type, $caption, $file_data) {
    global $pdo;
    
    logger()->info("پردازش آپلود فایل", [
        'chat_id' => $chat_id,
        'inline_button_id' => $inline_button_id,
        'file_type' => $file_type,
        'caption' => $caption,
        'function' => 'processFileUpload'
    ]);
    
    // تولید نام فایل منحصر به فرد
    $timestamp = time();
    $original_filename = $file_data['name'];
    $safe_filename = $timestamp . '_' . preg_replace('/[^a-zA-Z0-9\._-]/', '', $original_filename);
    $upload_path = 'uploads/' . $safe_filename;
    
    // اطمینان از وجود پوشه آپلود
    if (!file_exists('uploads')) {
        mkdir('uploads', 0755, true);
        logger()->debug("پوشه آپلود ایجاد شد", ['path' => 'uploads']);
    }
    
    // آپلود فایل
    if ($file_type === 'text') {
        // ذخیره فایل متنی
        if (file_put_contents($upload_path, $file_data['content']) === false) {
            logger()->error("خطا در آپلود فایل متنی", [
                'upload_path' => $upload_path,
                'error' => error_get_last()
            ]);
            return false;
        }
    } else {
        // ذخیره فایل از طریق آپلود
        if (!move_uploaded_file($file_data['tmp_name'], $upload_path)) {
            logger()->error("خطا در آپلود فایل", [
                'upload_path' => $upload_path,
                'error' => error_get_last()
            ]);
            return false;
        }
    }
    
    logger()->debug("فایل با موفقیت آپلود شد", [
        'upload_path' => $upload_path,
        'file_type' => $file_type
    ]);
    
    // ثبت فایل در پایگاه داده
    $stmt = $pdo->prepare("INSERT INTO files (inline_button_id, file_type, file_path, caption) VALUES (:ibid, :type, :path, :caption)");
    $result = $stmt->execute([
        ':ibid' => $inline_button_id,
        ':type' => $file_type,
        ':path' => $upload_path,
        ':caption' => $caption
    ]);
    
    if (!$result) {
        logger()->error("خطا در ثبت فایل در پایگاه داده", [
            'error' => $stmt->errorInfo()
        ]);
        // حذف فایل آپلود شده در صورت خطا
        @unlink($upload_path);
        return false;
    }
    
    logger()->info("فایل با موفقیت در پایگاه داده ثبت شد", [
        'file_id' => $pdo->lastInsertId(),
        'inline_button_id' => $inline_button_id,
        'file_path' => $upload_path
    ]);
    
    return true;
}

function showMainMenu($chat_id) {
    $keyboard = [
        ['آزمون‌های فعال'],
        ['راهنما', 'درباره ما']
    ];
    sendMessage($chat_id, "به منوی اصلی خوش آمدید!", [
        'reply_markup' => json_encode(['keyboard' => $keyboard, 'resize_keyboard' => true])
    ]);
}
?>
